//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cTextureManager.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cTextureManager.hpp"

#include <stdexcept>
#include <GL/gl.h>
#include <GL/glu.h>
#include <SDL_image.h>
#include "Debug.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================

//==============================================================================
//! Constructor
//------------------------------------------------------------------------------
cTextureManager::cTextureManager (void):
mDataDir ("")
{
	// Empty
};
//==============================================================================

//==============================================================================
//! Destructor
//------------------------------------------------------------------------------
cTextureManager::~cTextureManager (void)
{
	FreeAllTextures ();
};
//==============================================================================

//==============================================================================
//! Return singleton
//------------------------------------------------------------------------------
cTextureManager &
cTextureManager::GetInstance (void)
{
	static cTextureManager singleton;
	return singleton;
}
//==============================================================================

//==============================================================================
//! Load texture from a file
//------------------------------------------------------------------------------
Uint32 
cTextureManager::LoadTexture (string filename)
{
	if ( mTextures[filename].ID != 0 )
		return mTextures[filename].ID;
	
	string temp = mDataDir;
	temp.append (filename);
	
	dbgInfo () << "Loading texture from file " << temp << '\n';
	
	// Load image
	SDL_Surface *pImage = IMG_Load (temp.c_str ());
	if ( pImage == NULL )
	{
		dbgError () << "Unable to load texture: " << IMG_GetError () << '\n';
		return 0;
	}
	
	tTextureInfo info;
	
	// Make texture
	info.ID = MakeTexture (pImage);
	info.width = pImage->w;
	info.height = pImage->h;
	
	// Free the image
	SDL_FreeSurface (pImage);
	
	if ( info.ID != 0 )
	{
		// Store the ID
		mTextures[filename] = info;
	}	
	return info.ID;
}
//==============================================================================

//! Get texture size
void 
cTextureManager::GetTextureSize (string filename, int &w, int &h)
{
	if ( mTextures[filename].ID == 0 )
		throw runtime_error ("Texture size query for unloaded texture");
	
	w = mTextures[filename].width;
	h = mTextures[filename].height;
}

//==============================================================================	
//! Free texture
//------------------------------------------------------------------------------
void 
cTextureManager::FreeTexture (string filename)
{
	Uint32 ID = mTextures[filename].ID;
	if ( ID == 0 )
		return;
	
	dbgInfo () << "Freeing texture " << filename << " ID: " << ID << '\n';
	
	glDeleteTextures (1, &ID);
	mTextures.erase (filename);
}
//==============================================================================

//==============================================================================
//! Free all textures
//------------------------------------------------------------------------------
void 
cTextureManager::FreeAllTextures (void)
{
	if ( mTextures.empty () )
		return;

	dbgInfo () << "Freeing all textures\n";
	
	Uint32 ID;
	map<string, tTextureInfo>::iterator textureID = mTextures.begin ();
	while ( textureID != mTextures.end () )
	{
		ID = textureID->second.ID;
		glDeleteTextures (1, &ID);
		textureID++;
	}
	mTextures.clear ();
}
//==============================================================================
	
//==============================================================================
//! Make texture from a SDL_Surface
//------------------------------------------------------------------------------
Uint32 
cTextureManager::MakeTexture (SDL_Surface *pTexture)
{
	dbg::check_ptr (dbg::error, pTexture, DBG_HERE);
	
	GLint internalFormat = GL_RGB;
	GLenum format;
	
	if ( pTexture->format->Rshift < pTexture->format->Gshift )
	{
		if ( pTexture->format->Ashift != 0 )
		{
			dbgInfo () << "Alpha channel found! Using RGBA as internal texture format\n";
			format = GL_RGBA;
			internalFormat = GL_RGBA;
		}
		else
			format = GL_RGB;
	}
	else
		format = GL_BGR;	
	
	// Generate texture
	GLuint textureID;
	glGenTextures (1, &textureID);
	
	glBindTexture (GL_TEXTURE_2D, textureID);
	
	// We don't need mipmaps
	glTexImage2D (GL_TEXTURE_2D, 0, internalFormat, pTexture->w, pTexture->h, 0, format, GL_UNSIGNED_BYTE, pTexture->pixels);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
	
	// Generate mipmaps
	//gluBuild2DMipmaps (GL_TEXTURE_2D, internalFormat, pTexture->w, pTexture->h, format, GL_UNSIGNED_BYTE, pTexture->pixels);		
	//glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST_MIPMAP_LINEAR);
	//glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST_MIPMAP_LINEAR);
	
	GLenum error = glGetError ();
	while ( error != GL_NO_ERROR )
	{
		dbgError () << "cTextureManager: OpenGL error: " << gluErrorString (error) << '\n';
		error = glGetError ();
	}
	
	return textureID;
}
//==============================================================================

//==============================================================================
// EOF
//==============================================================================
