// This file Copyright © Mnemosyne LLC.
// It may be used under GPLv2 (SPDX: GPL-2.0-only), GPLv3 (SPDX: GPL-3.0-only),
// or any future license endorsed by Mnemosyne LLC.
// License text can be found in the licenses/ folder.

#include <algorithm>
#include <array>
#include <cctype> // isspace
#include <cmath> // floor
#include <chrono>
#include <cstdint> // int64_t
#include <cstdio>
#include <cstdlib>
#include <cstring> /* strcmp */
#include <ctime>
#include <limits>
#include <optional>
#include <set>
#include <string>
#include <string_view>
#include <utility>
#include <vector>

#include <curl/curl.h>

#include <fmt/chrono.h>
#include <fmt/format.h>

#include <libtransmission/transmission.h>

#include <libtransmission/api-compat.h>
#include <libtransmission/crypto-utils.h>
#include <libtransmission/file.h>
#include <libtransmission/quark.h>
#include <libtransmission/rpcimpl.h>
#include <libtransmission/tr-assert.h>
#include <libtransmission/tr-getopt.h>
#include <libtransmission/utils.h>
#include <libtransmission/values.h>
#include <libtransmission/variant.h>
#include <libtransmission/version.h>

using namespace std::literals;

namespace api_compat = libtransmission::api_compat;
using namespace libtransmission::Values;

#define SPEED_K_STR "kB/s"
#define MEM_M_STR "MiB"

namespace
{
auto constexpr DefaultPort = uint16_t{ TrDefaultRpcPort };
char constexpr DefaultHost[] = "localhost";
char constexpr DefaultUrl[] = TR_DEFAULT_RPC_URL_STR "rpc/";

char constexpr MyName[] = "transmission-remote";
char constexpr Usage[] = "transmission-remote " LONG_VERSION_STRING
                         "\n"
                         "A fast and easy BitTorrent client\n"
                         "https://transmissionbt.com/\n"
                         "\n"
                         "Usage: transmission-remote [host] [options]\n"
                         "       transmission-remote [port] [options]\n"
                         "       transmission-remote [host:port] [options]\n"
                         "       transmission-remote [http(s?)://host:port/transmission/] [options]\n"
                         "\n"
                         "See the man page for detailed explanations and many examples.";

struct RemoteConfig
{
    std::string auth;
    std::string filter;
    std::string netrc;
    std::string session_id;
    std::string torrent_ids;
    std::string unix_socket_path;

    bool debug = false;
    bool json = false;
    bool use_ssl = false;

    api_compat::Style network_style = api_compat::Style::Tr4;
};

// --- Display Utilities

[[nodiscard]] std::string eta_to_string(int64_t eta)
{
    if (eta < 0)
    {
        return "Unknown"s;
    }

    if (eta < 60)
    {
        return fmt::format("{:d} sec", eta);
    }

    if (eta < (60 * 60))
    {
        return fmt::format("{:d} min", eta / 60);
    }

    if (eta < (60 * 60 * 24))
    {
        return fmt::format("{:d} hrs", eta / (60 * 60));
    }

    if (eta < (60 * 60 * 24 * 30))
    {
        return fmt::format("{:d} days", eta / (60 * 60 * 24));
    }

    if (eta < (60 * 60 * 24 * 30 * 12))
    {
        return fmt::format("{:d} months", eta / (60 * 60 * 24 * 30));
    }

    if (eta < (60 * 60 * 24 * 365 * 1000LL)) // up to 999 years
    {
        return fmt::format("{:d} years", eta / (60 * 60 * 24 * 365));
    }

    return "∞"s;
}

[[nodiscard]] auto tr_strltime(time_t seconds)
{
    if (seconds < 0)
    {
        seconds = 0;
    }

    auto const total_seconds = seconds;
    auto const days = seconds / 86400;
    auto const hours = (seconds % 86400) / 3600;
    auto const minutes = (seconds % 3600) / 60;
    seconds = (seconds % 3600) % 60;

    auto tmpstr = std::string{};

    auto const hstr = fmt::format("{:d} {:s}", hours, tr_ngettext("hour", "hours", hours));
    auto const mstr = fmt::format("{:d} {:s}", minutes, tr_ngettext("minute", "minutes", minutes));
    auto const sstr = fmt::format("{:d} {:s}", seconds, tr_ngettext("seconds", "seconds", seconds));

    if (days > 0)
    {
        auto const dstr = fmt::format("{:d} {:s}", days, tr_ngettext("day", "days", days));
        tmpstr = days >= 4 || hours == 0 ? dstr : fmt::format("{:s}, {:s}", dstr, hstr);
    }
    else if (hours > 0)
    {
        tmpstr = hours >= 4 || minutes == 0 ? hstr : fmt::format("{:s}, {:s}", hstr, mstr);
    }
    else if (minutes > 0)
    {
        tmpstr = minutes >= 4 || seconds == 0 ? mstr : fmt::format("{:s}, {:s}", mstr, sstr);
    }
    else
    {
        tmpstr = sstr;
    }

    auto const totstr = fmt::format("{:d} {:s}", total_seconds, tr_ngettext("seconds", "seconds", total_seconds));
    return fmt::format("{:s} ({:s})", tmpstr, totstr);
}

[[nodiscard]] auto strlpercent(double x)
{
    return tr_strpercent(x);
}

[[nodiscard]] auto strlratio2(double ratio)
{
    return tr_strratio(ratio, "None", "Inf");
}

[[nodiscard]] auto strlratio(int64_t numerator, int64_t denominator)
{
    return strlratio2(tr_getRatio(numerator, denominator));
}

[[nodiscard]] auto strlsize(int64_t bytes)
{
    if (bytes < 0)
    {
        return "Unknown"s;
    }

    if (bytes == 0)
    {
        return "None"s;
    }

    return Storage{ bytes, Storage::Units::Bytes }.to_string();
}

enum
{
    ID_NOOP,
    ID_SESSION,
    ID_STATS,
    ID_DETAILS,
    ID_FILES,
    ID_FILTER,
    ID_GROUPS,
    ID_LIST,
    ID_PEERS,
    ID_PIECES,
    ID_PORTTEST,
    ID_TORRENT_ADD,
    ID_TRACKERS,
    ID_BLOCKLIST,
};

// --- Command-Line Arguments

using Arg = tr_option::Arg;
auto constexpr Options = std::array<tr_option, 106>{ {
    { 'a', "add", "Add torrent files by filename or URL", "a", Arg::None, nullptr },
    { 970, "alt-speed", "Use the alternate Limits", "as", Arg::None, nullptr },
    { 971, "no-alt-speed", "Don't use the alternate Limits", "AS", Arg::None, nullptr },
    { 972, "alt-speed-downlimit", "max alternate download speed (in " SPEED_K_STR ")", "asd", Arg::Required, "<speed>" },
    { 973, "alt-speed-uplimit", "max alternate upload speed (in " SPEED_K_STR ")", "asu", Arg::Required, "<speed>" },
    { 974, "alt-speed-scheduler", "Use the scheduled on/off times", "asc", Arg::None, nullptr },
    { 975, "no-alt-speed-scheduler", "Don't use the scheduled on/off times", "ASC", Arg::None, nullptr },
    { 976, "alt-speed-time-begin", "Time to start using the alt speed limits (in hhmm)", nullptr, Arg::Required, "<time>" },
    { 977, "alt-speed-time-end", "Time to stop using the alt speed limits (in hhmm)", nullptr, Arg::Required, "<time>" },
    { 978, "alt-speed-days", "Numbers for any/all days of the week - eg. \"1-7\"", nullptr, Arg::Required, "<days>" },
    { 963, "blocklist-update", "Blocklist update", nullptr, Arg::None, nullptr },
    { 'c', "incomplete-dir", "Where to store new torrents until they're complete", "c", Arg::Required, "<dir>" },
    { 'C', "no-incomplete-dir", "Don't store incomplete torrents in a different location", "C", Arg::None, nullptr },
    { 'b', "debug", "Print debugging information", "b", Arg::None, nullptr },
    { 730, "bandwidth-group", "Set the current torrents' bandwidth group", "bwg", Arg::Required, "<group>" },
    { 731, "no-bandwidth-group", "Reset the current torrents' bandwidth group", "nwg", Arg::None, nullptr },
    { 732, "list-groups", "Show bandwidth groups with their parameters", "lg", Arg::None, nullptr },
    { 'd',
      "downlimit",
      "Set the max download speed in " SPEED_K_STR " for the current torrent(s) or globally",
      "d",
      Arg::Required,
      "<speed>" },
    { 'D', "no-downlimit", "Disable max download speed for the current torrent(s) or globally", "D", Arg::None, nullptr },
    { 'e', "cache", "Set the maximum size of the session's memory cache (in " MEM_M_STR ")", "e", Arg::Required, "<size>" },
    { 910, "encryption-required", "Encrypt all peer connections", "er", Arg::None, nullptr },
    { 911, "encryption-preferred", "Prefer encrypted peer connections", "ep", Arg::None, nullptr },
    { 912, "encryption-tolerated", "Prefer unencrypted peer connections", "et", Arg::None, nullptr },
    { 850, "exit", "Tell the transmission session to shut down", nullptr, Arg::None, nullptr },
    { 940, "files", "List the current torrent(s)' files", "f", Arg::None, nullptr },
    { 'F', "filter", "Filter the current torrent(s)", "F", Arg::Required, "criterion" },
    { 'g', "get", "Mark files for download", "g", Arg::Required, "<files>" },
    { 'G', "no-get", "Mark files for not downloading", "G", Arg::Required, "<files>" },
    { 'i', "info", "Show the current torrent(s)' details", "i", Arg::None, nullptr },
    { 944, "print-ids", "Print the current torrent(s)' ids", "ids", Arg::None, nullptr },
    { 940, "info-files", "List the current torrent(s)' files", "if", Arg::None, nullptr },
    { 941, "info-peers", "List the current torrent(s)' peers", "ip", Arg::None, nullptr },
    { 942, "info-pieces", "List the current torrent(s)' pieces", "ic", Arg::None, nullptr },
    { 943, "info-trackers", "List the current torrent(s)' trackers", "it", Arg::None, nullptr },
    { 'j', "json", "Return RPC response as a JSON string", "j", Arg::None, nullptr },
    { 920, "session-info", "Show the session's details", "si", Arg::None, nullptr },
    { 921, "session-stats", "Show the session's statistics", "st", Arg::None, nullptr },
    { 'l', "list", "List all torrents", "l", Arg::None, nullptr },
    { 'L', "labels", "Set the current torrents' labels", "L", Arg::Required, "<label[,label...]>" },
    { 960, "move", "Move current torrent's data to a new folder", nullptr, Arg::Required, "<path>" },
    { 968, "unix-socket", "Use a Unix domain socket", nullptr, Arg::Required, "<path>" },
    { 961, "find", "Tell Transmission where to find a torrent's data", nullptr, Arg::Required, "<path>" },
    { 964, "rename", "Rename torrents root folder or a file", nullptr, Arg::Required, "<name>" },
    { 965, "path", "Provide path for rename functions", nullptr, Arg::Required, "<path>" },
    { 'm', "portmap", "Enable portmapping via NAT-PMP or UPnP", "m", Arg::None, nullptr },
    { 'M', "no-portmap", "Disable portmapping", "M", Arg::None, nullptr },
    { 'n', "auth", "Set username and password", "n", Arg::Required, "<user:pw>" },
    { 810, "authenv", "Set authentication info from the TR_AUTH environment variable (user:pw)", "ne", Arg::None, nullptr },
    { 'N', "netrc", "Set authentication info from a .netrc file", "N", Arg::Required, "<file>" },
    { 820, "ssl", "Use SSL when talking to daemon", nullptr, Arg::None, nullptr },
    { 'o', "dht", "Enable distributed hash tables (DHT)", "o", Arg::None, nullptr },
    { 'O', "no-dht", "Disable distributed hash tables (DHT)", "O", Arg::None, nullptr },
    { 'p', "port", "Port for incoming peers (Default: " TR_DEFAULT_PEER_PORT_STR ")", "p", Arg::Required, "<port>" },
    { 962, "port-test", "Port testing", "pt", Arg::None, nullptr },
    { 'P', "random-port", "Random port for incoming peers", "P", Arg::None, nullptr },
    { 900, "priority-high", "Try to download these file(s) first", "ph", Arg::Required, "<files>" },
    { 901, "priority-normal", "Try to download these file(s) normally", "pn", Arg::Required, "<files>" },
    { 902, "priority-low", "Try to download these file(s) last", "pl", Arg::Required, "<files>" },
    { 700, "bandwidth-high", "Give this torrent first chance at available bandwidth", "Bh", Arg::None, nullptr },
    { 701, "bandwidth-normal", "Give this torrent bandwidth left over by high priority torrents", "Bn", Arg::None, nullptr },
    { 702,
      "bandwidth-low",
      "Give this torrent bandwidth left over by high and normal priority torrents",
      "Bl",
      Arg::None,
      nullptr },
    { 600, "reannounce", "Reannounce the current torrent(s)", nullptr, Arg::None, nullptr },
    { 'r', "remove", "Remove the current torrent(s)", "r", Arg::None, nullptr },
    { 930, "peers", "Set the maximum number of peers for the current torrent(s) or globally", "pr", Arg::Required, "<max>" },
    { 840, "remove-and-delete", "Remove the current torrent(s) and delete local data", "rad", Arg::None, nullptr },
    { 800, "torrent-done-script", "A script to run when a torrent finishes downloading", nullptr, Arg::Required, "<file>" },
    { 801, "no-torrent-done-script", "Don't run the done-downloading script", nullptr, Arg::None, nullptr },
    { 802, "torrent-done-seeding-script", "A script to run when a torrent finishes seeding", nullptr, Arg::Required, "<file>" },
    { 803, "no-torrent-done-seeding-script", "Don't run the done-seeding script", nullptr, Arg::None, nullptr },
    { 950, "seedratio", "Let the current torrent(s) seed until a specific ratio", "sr", Arg::Required, "ratio" },
    { 951, "seedratio-default", "Let the current torrent(s) use the global seedratio settings", "srd", Arg::None, nullptr },
    { 952, "no-seedratio", "Let the current torrent(s) seed regardless of ratio", "SR", Arg::None, nullptr },
    { 953,
      "global-seedratio",
      "All torrents, unless overridden by a per-torrent setting, should seed until a specific ratio",
      "gsr",
      Arg::Required,
      "ratio" },
    { 954,
      "no-global-seedratio",
      "All torrents, unless overridden by a per-torrent setting, should seed regardless of ratio",
      "GSR",
      Arg::None,
      nullptr },
    { 955,
      "idle-seeding-limit",
      "Let the current torrent(s) seed until a specific amount idle time",
      "isl",
      Arg::Required,
      "<minutes>" },
    { 956,
      "default-idle-seeding-limit",
      "Let the current torrent(s) use the default idle seeding settings",
      "isld",
      Arg::None,
      nullptr },
    { 957, "no-idle-seeding-limit", "Let the current torrent(s) seed regardless of idle time", "ISL", Arg::None, nullptr },
    { 958,
      "global-idle-seeding-limit",
      "All torrents, unless overridden by a per-torrent setting, should seed until a specific amount of idle time",
      "gisl",
      Arg::Required,
      "<minutes>" },
    { 959,
      "no-global-idle-seeding-limit",
      "All torrents, unless overridden by a per-torrent setting, should seed regardless of idle time",
      "GISL",
      Arg::None,
      nullptr },
    { 710, "tracker-add", "Add a tracker to a torrent", "td", Arg::Required, "<tracker>" },
    { 712, "tracker-remove", "Remove a tracker from a torrent", "tr", Arg::Required, "<trackerId>" },
    { 's', "start", "Start the current torrent(s)", "s", Arg::None, nullptr },
    { 'S', "stop", "Stop the current torrent(s)", "S", Arg::None, nullptr },
    { 't', "torrent", "Set the current torrent(s)", "t", Arg::Required, "<torrent>" },
    { 990, "start-paused", "Start added torrents paused", nullptr, Arg::None, nullptr },
    { 991, "no-start-paused", "Start added torrents unpaused", nullptr, Arg::None, nullptr },
    { 992, "trash-torrent", "Delete torrents after adding", nullptr, Arg::None, nullptr },
    { 993, "no-trash-torrent", "Do not delete torrents after adding", nullptr, Arg::None, nullptr },
    { 994,
      "sequential-download",
      "Download the torrent sequentially, starting from <piece> or 0",
      "seq",
      Arg::Optional,
      "<piece>" },
    { 995, "no-sequential-download", "Download the torrent normally", "SEQ", Arg::None, nullptr },
    { 984, "honor-session", "Make the current torrent(s) honor the session limits", "hl", Arg::None, nullptr },
    { 985, "no-honor-session", "Make the current torrent(s) not honor the session limits", "HL", Arg::None, nullptr },
    { 'u',
      "uplimit",
      "Set the max upload speed in " SPEED_K_STR " for the current torrent(s) or globally",
      "u",
      Arg::Required,
      "<speed>" },
    { 'U', "no-uplimit", "Disable max upload speed for the current torrent(s) or globally", "U", Arg::None, nullptr },
    { 830,
      "preferred-transports",
      "Comma-separated list specifying preference of transport protocols",
      nullptr,
      Arg::Required,
      "<protocol(s)>" },
    { 831, "utp", "*DEPRECATED* Enable µTP for peer connections", nullptr, Arg::None, nullptr },
    { 832, "no-utp", "*DEPRECATED* Disable µTP for peer connections", nullptr, Arg::None, nullptr },
    { 'v', "verify", "Verify the current torrent(s)", "v", Arg::None, nullptr },
    { 'V', "version", "Show version number and exit", "V", Arg::None, nullptr },
    { 'w',
      "download-dir",
      "When used in conjunction with --add, set the new torrent's download folder. "
      "Otherwise, set the default download folder",
      "w",
      Arg::Required,
      "<path>" },
    { 'x', "pex", "Enable peer exchange (PEX)", "x", Arg::None, nullptr },
    { 'X', "no-pex", "Disable peer exchange (PEX)", "X", Arg::None, nullptr },
    { 'y', "lpd", "Enable local peer discovery (LPD)", "y", Arg::None, nullptr },
    { 'Y', "no-lpd", "Disable local peer discovery (LPD)", "Y", Arg::None, nullptr },
    { 941, "peer-info", "List the current torrent(s)' peers", "pi", Arg::None, nullptr },
    { 0, nullptr, nullptr, nullptr, Arg::None, nullptr },
} };
static_assert(Options[std::size(Options) - 2].val != 0);
} // namespace

namespace
{
void show_usage()
{
    tr_getopt_usage(MyName, Usage, std::data(Options));
}

[[nodiscard]] auto numarg(std::string_view arg)
{
    auto remainder = std::string_view{};
    auto const num = tr_num_parse<int64_t>(arg, &remainder);
    if (!num || !std::empty(remainder))
    {
        fmt::print(stderr, "Not a number: '{:s}'\n", arg);
        show_usage();
        exit(EXIT_FAILURE);
    }

    return *num;
}

enum
{
    MODE_META_COMMAND = 0,
    MODE_TORRENT_ACTION = 1 << 0,
    MODE_TORRENT_ADD = 1 << 1,
    MODE_TORRENT_GET = 1 << 2,
    MODE_TORRENT_REMOVE = 1 << 3,
    MODE_TORRENT_SET = 1 << 4,
    MODE_TORRENT_START_STOP = 1 << 5,
    MODE_SESSION_SET = 1 << 6
};

[[nodiscard]] int get_opt_mode(int val)
{
    switch (val)
    {
    case TR_OPT_ERR:
    case TR_OPT_UNK:
    case 'a': /* add torrent */
    case 'b': /* debug */
    case 'n': /* auth */
    case 968: /* Unix domain socket */
    case 'j': /* JSON */
    case 810: /* authenv */
    case 'N': /* netrc */
    case 820: /* UseSSL */
    case 't': /* set current torrent */
    case 'V': /* show version number */
    case 944: /* print selected torrents' ids */
        return MODE_META_COMMAND;

    case 'c': /* incomplete-dir */
    case 'C': /* no-incomplete-dir */
    case 'e': /* cache */
    case 'm': /* portmap */
    case 'M': /* no-portmap */
    case 'o': /* dht */
    case 'O': /* no-dht */
    case 'p': /* incoming peer port */
    case 'P': /* random incoming peer port */
    case 'x': /* pex */
    case 'X': /* no-pex */
    case 'y': /* lpd */
    case 'Y': /* no-lpd */
    case 800: /* torrent-done-script */
    case 801: /* no-torrent-done-script */
    case 802: /* torrent-done-seeding-script */
    case 803: /* no-torrent-done-seeding-script */
    case 830: /* preferred-transports */
    case 831: /* utp */
    case 832: /* no-utp */
    case 970: /* alt-speed */
    case 971: /* no-alt-speed */
    case 972: /* alt-speed-downlimit */
    case 973: /* alt-speed-uplimit */
    case 974: /* alt-speed-scheduler */
    case 975: /* no-alt-speed-scheduler */
    case 976: /* alt-speed-time-begin */
    case 977: /* alt-speed-time-end */
    case 978: /* alt-speed-days */
    case 910: /* encryption-required */
    case 911: /* encryption-preferred */
    case 912: /* encryption-tolerated */
    case 953: /* global-seedratio */
    case 954: /* no-global-seedratio */
    case 958: /* global-idle-seeding-limit */
    case 959: /* no-global-idle-seeding-limit */
    case 990: /* start-paused */
    case 991: /* no-start-paused */
    case 992: /* trash-torrent */
    case 993: /* no-trash-torrent */
        return MODE_SESSION_SET;

    case 712: /* tracker-remove */
    case 950: /* seedratio */
    case 951: /* seedratio-default */
    case 952: /* no-seedratio */
    case 955: /* idle-seeding-limit */
    case 956: /* default-idle-seeding-limit */
    case 957: /* no-idle-seeding-limit*/
    case 984: /* honor-session */
    case 985: /* no-honor-session */
        return MODE_TORRENT_SET;

    case 'g': /* get */
    case 'G': /* no-get */
    case 'L': /* labels */
    case 700: /* torrent priority-high */
    case 701: /* torrent priority-normal */
    case 702: /* torrent priority-low */
    case 710: /* tracker-add */
    case 900: /* file priority-high */
    case 901: /* file priority-normal */
    case 902: /* file priority-low */
    case 730: /* set bandwidth group */
    case 731: /* reset bandwidth group */
        return MODE_TORRENT_SET | MODE_TORRENT_ADD;

    case 'i': /* info */
    case 'l': /* list all torrents */
    case 940: /* info-files */
    case 941: /* info-peer */
    case 942: /* info-pieces */
    case 943: /* info-tracker */
    case 'F': /* filter torrents */
        return MODE_TORRENT_GET;

    case 'd': /* download speed limit */
    case 'D': /* no download speed limit */
    case 'u': /* upload speed limit */
    case 'U': /* no upload speed limit */
    case 930: /* peers */
        return MODE_SESSION_SET | MODE_TORRENT_SET;

    case 994: /* sequential-download */
    case 995: /* no-sequential-download */
        return MODE_SESSION_SET | MODE_TORRENT_SET | MODE_TORRENT_ADD;

    case 'r': /* remove */
    case 840: /* remove and delete */
        return MODE_TORRENT_REMOVE;

    case 's': /* start */
    case 'S': /* stop */
        return MODE_TORRENT_START_STOP;

    case 'v': /* verify */
    case 600: /* reannounce */
        return MODE_TORRENT_ACTION;

    case 'w': /* download-dir */
    case 850: /* session-close */
    case 732: /* List groups */
    case 920: /* session-info */
    case 921: /* session-stats */
    case 960: /* move */
    case 961: /* find */
    case 962: /* port-test */
    case 963: /* blocklist-update */
    case 964: /* rename */
    case 965: /* path */
        return -1;

    default:
        fmt::print(stderr, "unrecognized argument {:d}\n", val);
        TR_ASSERT_MSG(false, "unrecognized argument");
        return -2;
    }
}

[[nodiscard]] std::string get_encoded_metainfo(char const* filename)
{
    if (auto contents = std::vector<char>{}; tr_sys_path_exists(filename) && tr_file_read(filename, contents))
    {
        return tr_base64_encode({ std::data(contents), std::size(contents) });
    }

    return {};
}

/**
 * - values that are all-digits are numbers
 * - values that are all-digits or commas are number lists
 * - anything else is a string
 */
[[nodiscard]] tr_variant rpc_parse_list_str(std::string_view str)
{
    auto const values = tr_num_parse_range(str);
    auto const n_values = std::size(values);

    if (n_values == 0)
    {
        return { str };
    }

    if (n_values == 1)
    {
        return { values[0] };
    }

    auto num_vec = tr_variant::Vector{};
    num_vec.resize(n_values);
    std::copy_n(std::cbegin(values), n_values, std::begin(num_vec));
    return { std::move(num_vec) };
}

void add_id_arg(tr_variant::Map& params, std::string_view id_str, std::string_view fallback = "")
{
    if (std::empty(id_str))
    {
        id_str = fallback;
    }

    if (std::empty(id_str))
    {
        fmt::print(stderr, "No torrent specified!  Please use the -t option first.\n");
        id_str = "-1"sv; // no torrent will have this ID, so will act as a no-op
    }

    static auto constexpr IdActive = "active"sv;
    static auto constexpr IdAll = "all"sv;

    if (IdActive == id_str)
    {
        params.insert_or_assign(TR_KEY_ids, tr_variant::unmanaged_string(TR_KEY_recently_active));
    }
    else if (IdAll != id_str)
    {
        bool const is_list = id_str.find_first_of(",-") != std::string_view::npos;
        bool is_num = true;

        for (auto const& ch : id_str)
        {
            is_num = is_num && isdigit(ch);
        }

        if (is_num || is_list)
        {
            params.insert_or_assign(TR_KEY_ids, rpc_parse_list_str(id_str));
        }
        else
        {
            params.insert_or_assign(TR_KEY_ids, id_str); /* it's a torrent sha hash */
        }
    }
}

void add_id_arg(tr_variant::Map& args, RemoteConfig const& config, std::string_view fallback = "")
{
    return add_id_arg(args, config.torrent_ids, fallback);
}

void add_time(tr_variant::Map& args, tr_quark const key, std::string_view arg)
{
    if (std::size(arg) == 4)
    {
        auto const hour = tr_num_parse<int>(arg.substr(0, 2)).value_or(-1);
        auto const min = tr_num_parse<int>(arg.substr(2, 2)).value_or(-1);

        if (0 <= hour && hour < 24 && 0 <= min && min < 60)
        {
            args.insert_or_assign(key, min + hour * 60);
            return;
        }
    }

    fmt::print(stderr, "Please specify the time of day in 'hhmm' format.\n");
}

void add_days(tr_variant::Map& args, tr_quark const key, std::string_view arg)
{
    int days = 0;

    if (!std::empty(arg))
    {
        for (int& day : tr_num_parse_range(arg))
        {
            if (day < 0 || day > 7)
            {
                continue;
            }

            if (day == 7)
            {
                day = 0;
            }

            days |= 1 << day;
        }
    }

    if (days != 0)
    {
        args.insert_or_assign(key, days);
    }
    else
    {
        fmt::print(stderr, "Please specify the days of the week in '1-3,4,7' format.\n");
    }
}

void add_labels(tr_variant::Map& args, std::string_view comma_delimited_labels)
{
    auto* labels = args.find_if<tr_variant::Vector>(TR_KEY_labels);
    if (labels == nullptr)
    {
        labels = args.insert_or_assign(TR_KEY_labels, tr_variant::make_vector(10)).first.get_if<tr_variant::Vector>();
    }

    auto label = std::string_view{};
    while (tr_strv_sep(&comma_delimited_labels, &label, ','))
    {
        labels->emplace_back(label);
    }
}

void set_group(tr_variant::Map& args, std::string_view group)
{
    args.insert_or_assign(TR_KEY_group, tr_variant::unmanaged_string(group));
}

void set_preferred_transports(tr_variant::Map& args, std::string_view comma_delimited_protocols)
{
    auto* preferred_protocols = args.find_if<tr_variant::Vector>(TR_KEY_preferred_transports);
    if (preferred_protocols == nullptr)
    {
        preferred_protocols = args.insert_or_assign(TR_KEY_preferred_transports, tr_variant::make_vector(10))
                                  .first.get_if<tr_variant::Vector>();
    }

    auto protocol = std::string_view{};
    while (tr_strv_sep(&comma_delimited_protocols, &protocol, ','))
    {
        preferred_protocols->emplace_back(protocol);
    }
}

[[nodiscard]] auto make_files_list(std::string_view str_in)
{
    if (std::empty(str_in))
    {
        fmt::print(stderr, "No files specified!\n");
        str_in = "-1"sv; // no file will have this index, so should be a no-op
    }

    auto files = tr_variant::Vector{};

    if (str_in != "all"sv)
    {
        files.reserve(100U);
        for (auto const& idx : tr_num_parse_range(str_in))
        {
            files.emplace_back(idx);
        }
    }

    return files;
}

auto constexpr FilesKeys = std::array<tr_quark, 4>{
    TR_KEY_files,
    TR_KEY_name,
    TR_KEY_priorities,
    TR_KEY_wanted,
};
static_assert(FilesKeys[std::size(FilesKeys) - 1] != tr_quark{});

auto constexpr DetailsKeys = std::array<tr_quark, 57>{
    TR_KEY_activity_date,
    TR_KEY_added_date,
    TR_KEY_bandwidth_priority,
    TR_KEY_comment,
    TR_KEY_corrupt_ever,
    TR_KEY_creator,
    TR_KEY_date_created,
    TR_KEY_desired_available,
    TR_KEY_done_date,
    TR_KEY_download_dir,
    TR_KEY_downloaded_ever,
    TR_KEY_download_limit,
    TR_KEY_download_limited,
    TR_KEY_error,
    TR_KEY_error_string,
    TR_KEY_eta,
    TR_KEY_group,
    TR_KEY_hash_string,
    TR_KEY_have_unchecked,
    TR_KEY_have_valid,
    TR_KEY_honors_session_limits,
    TR_KEY_id,
    TR_KEY_is_finished,
    TR_KEY_is_private,
    TR_KEY_labels,
    TR_KEY_left_until_done,
    TR_KEY_magnet_link,
    TR_KEY_name,
    TR_KEY_peers_connected,
    TR_KEY_peers_getting_from_us,
    TR_KEY_peers_sending_to_us,
    TR_KEY_peer_limit,
    TR_KEY_percent_done,
    TR_KEY_piece_count,
    TR_KEY_piece_size,
    TR_KEY_rate_download,
    TR_KEY_rate_upload,
    TR_KEY_recheck_progress,
    TR_KEY_seconds_downloading,
    TR_KEY_seconds_seeding,
    TR_KEY_seed_idle_mode,
    TR_KEY_seed_idle_limit,
    TR_KEY_seed_ratio_mode,
    TR_KEY_seed_ratio_limit,
    TR_KEY_sequential_download,
    TR_KEY_sequential_download_from_piece,
    TR_KEY_size_when_done,
    TR_KEY_source,
    TR_KEY_start_date,
    TR_KEY_status,
    TR_KEY_total_size,
    TR_KEY_uploaded_ever,
    TR_KEY_upload_limit,
    TR_KEY_upload_limited,
    TR_KEY_upload_ratio,
    TR_KEY_webseeds,
    TR_KEY_webseeds_sending_to_us,
};
static_assert(DetailsKeys[std::size(DetailsKeys) - 1] != tr_quark{});

auto constexpr ListKeys = std::array<tr_quark, 15U>{
    TR_KEY_added_date,
    TR_KEY_error,
    TR_KEY_error_string,
    TR_KEY_eta,
    TR_KEY_id,
    TR_KEY_is_finished,
    TR_KEY_left_until_done,
    TR_KEY_name,
    TR_KEY_peers_getting_from_us,
    TR_KEY_peers_sending_to_us,
    TR_KEY_rate_download,
    TR_KEY_rate_upload,
    TR_KEY_size_when_done,
    TR_KEY_status,
    TR_KEY_upload_ratio,
};
static_assert(ListKeys[std::size(ListKeys) - 1] != tr_quark{});

[[nodiscard]] size_t write_func(void* ptr, size_t size, size_t nmemb, void* vbuf)
{
    auto const n_bytes = size * nmemb;
    static_cast<std::string*>(vbuf)->append(static_cast<char const*>(ptr), n_bytes);
    return n_bytes;
}

namespace header_utils
{
// `${name}: {$value}` --> std::pair<name, value>
[[nodiscard]] std::optional<std::pair<std::string_view, std::string_view>> parse_header(std::string_view const line)
{
    static auto constexpr Delimiter = ": "sv;
    if (auto const pos = line.find(Delimiter); pos != std::string_view::npos)
    {
        auto const name = tr_strv_strip(line.substr(0, pos));
        auto const value = tr_strv_strip(line.substr(pos + std::size(Delimiter)));
        return std::make_pair(name, value);
    }
    return {};
}

void warn_if_unsupported_rpc_version(std::string_view const semver)
{
    static auto constexpr ExpectedMajor = TrRpcVersionSemverMajor;
    auto const major_str = semver.substr(0, semver.find('.'));
    if (auto const major = tr_num_parse<int>(major_str); major && *major > ExpectedMajor)
    {
        fmt::print(
            stderr,
            "Warning: Server RPC version is {:s}, which may be incompatible with our version {:s}.\n",
            semver,
            TrRpcVersionSemver);
    }
}
} // namespace header_utils

// look for a session id in the header
// in case the server gives back a 409
[[nodiscard]] size_t parse_response_header(void* ptr, size_t size, size_t nmemb, void* vconfig)
{
    using namespace header_utils;
    static auto const session_id_header = tr_strlower(TR_RPC_SESSION_ID_HEADER);
    static auto const rpc_version_header = tr_strlower(TR_RPC_RPC_VERSION_HEADER);

    auto& config = *static_cast<RemoteConfig*>(vconfig);

    auto const line = std::string_view{ static_cast<char const*>(ptr), size * nmemb };

    if (auto const parsed = parse_header(line))
    {
        auto const [key, val] = *parsed;
        auto const key_lower = tr_strlower(key);

        if (key_lower == session_id_header)
        {
            config.session_id = val;
        }
        else if (key_lower == rpc_version_header)
        {
            config.network_style = api_compat::Style::Tr5;
            warn_if_unsupported_rpc_version(val);
        }
    }

    return std::size(line);
}

[[nodiscard]] std::string get_status_string(tr_variant::Map const& t)
{
    auto const status = t.value_if<int64_t>(TR_KEY_status);
    if (!status)
    {
        return ""s;
    }

    switch (*status)
    {
    case TR_STATUS_DOWNLOAD_WAIT:
    case TR_STATUS_SEED_WAIT:
        return "Queued"s;

    case TR_STATUS_STOPPED:
        if (t.value_if<bool>(TR_KEY_is_finished).value_or(false))
        {
            return "Finished"s;
        }
        return "Stopped"s;

    case TR_STATUS_CHECK_WAIT:
        if (auto const percent = t.value_if<double>(TR_KEY_recheck_progress))
        {
            return fmt::format("Will Verify ({:.0f}%)", floor(*percent * 100.0));
        }
        return "Will Verify"s;

    case TR_STATUS_CHECK:
        if (auto const percent = t.value_if<double>(TR_KEY_recheck_progress))
        {
            return fmt::format("Verifying ({:.0f}%)", floor(*percent * 100.0));
        }
        return "Verifying"s;

    case TR_STATUS_DOWNLOAD:
    case TR_STATUS_SEED:
        if (auto const from_us = t.value_if<int64_t>(TR_KEY_peers_getting_from_us).value_or(0),
            to_us = t.value_if<int64_t>(TR_KEY_peers_sending_to_us).value_or(0);
            from_us != 0 && to_us != 0)
        {
            return "Up & Down"s;
        }
        else if (to_us != 0)
        {
            return "Downloading"s;
        }
        else if (from_us == 0)
        {
            return "Idle"s;
        }
        if (auto left_until_done = t.value_if<int64_t>(TR_KEY_left_until_done).value_or(0); left_until_done > 0)
        {
            return "Uploading"s;
        }
        return "Seeding"s;

    default:
        return "Unknown"s;
    }
}

auto constexpr BandwidthPriorityNames = std::array<std::string_view, 4>{
    "Low"sv,
    "Normal"sv,
    "High"sv,
    "Invalid"sv,
};
static_assert(!BandwidthPriorityNames[std::size(BandwidthPriorityNames) - 1].empty());

template<size_t N>
std::string_view format_date(std::array<char, N>& buf, time_t now)
{
    auto begin = std::data(buf);
    auto end = fmt::format_to_n(begin, N, "{:%a %b %d %T %Y}", *std::localtime(&now)).out;
    return { begin, static_cast<size_t>(end - begin) };
}

void print_details(tr_variant::Map const& result)
{
    auto* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    for (auto const& t_var : *torrents)
    {
        auto* t = t_var.get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        std::array<char, 512> buf = {};

        fmt::print("NAME\n");

        if (auto const i = t->value_if<int64_t>(TR_KEY_id))
        {
            fmt::print("  Id: {:d}\n", *i);
        }

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_name))
        {
            fmt::print("  Name: {:s}\n", *sv);
        }

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_hash_string))
        {
            fmt::print("  Hash: {:s}\n", *sv);
        }

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_magnet_link))
        {
            fmt::print("  Magnet: {:s}\n", *sv);
        }

        if (auto* l = t->find_if<tr_variant::Vector>(TR_KEY_labels); l != nullptr)
        {
            fmt::print("  Labels: ");

            for (auto it = std::begin(*l), begin = std::begin(*l), end = std::end(*l); it != end; ++it)
            {
                if (auto sv = it->value_if<std::string_view>(); sv)
                {
                    fmt::print("{:s}{:s}", it != begin ? ", " : "", *sv);
                }
            }

            fmt::print("\n");
        }

        if (auto sv = t->value_if<std::string_view>(TR_KEY_group).value_or(""sv); !std::empty(sv))
        {
            fmt::print("  Bandwidth group: {:s}\n", sv);
        }

        fmt::print("\n");

        fmt::print("TRANSFER\n");
        fmt::print("  State: {:s}\n", get_status_string(*t));

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_download_dir))
        {
            fmt::print("  Location: {:s}\n", *sv);
        }

        if (auto const b = t->value_if<bool>(TR_KEY_sequential_download))
        {
            fmt::print("  Sequential Download: {:s}\n", *b ? "Yes" : "No");
            if (auto i = t->value_if<int64_t>(TR_KEY_sequential_download_from_piece); i)
            {
                fmt::print("  Sequential Download from piece: {:d}\n", *i);
            }
        }

        if (auto const d = t->value_if<double>(TR_KEY_percent_done))
        {
            fmt::print("  Percent Done: {:s}%\n", strlpercent(100.0 * *d));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_eta); i)
        {
            fmt::print("  ETA: {:s}\n", tr_strltime(*i));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_rate_download))
        {
            fmt::print("  Download Speed: {:s}\n", Speed{ *i, Speed::Units::Byps }.to_string());
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_rate_upload))
        {
            fmt::print("  Upload Speed: {:s}\n", Speed{ *i, Speed::Units::Byps }.to_string());
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_have_unchecked), j = t->value_if<int64_t>(TR_KEY_have_valid); i && j)
        {
            fmt::print("  Have: {:s} ({:s} verified)\n", strlsize(*i + *j), strlsize(*j));
        }

        if (auto const oi = t->value_if<int64_t>(TR_KEY_size_when_done))
        {
            auto const i = *oi;
            if (i < 1)
            {
                fmt::print("  Availability: None\n");
            }
            else if (auto j = t->value_if<int64_t>(TR_KEY_desired_available), k = t->value_if<int64_t>(TR_KEY_left_until_done);
                     j && k)
            {
                fmt::print("  Availability: {:s}%\n", strlpercent(100.0 * (*j + i - *k) / i));
            }

            if (auto j = t->value_if<int64_t>(TR_KEY_total_size))
            {
                fmt::print("  Total size: {:s} ({:s} wanted)\n", strlsize(*j), strlsize(i));
            }
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_downloaded_ever))
        {
            if (auto corrupt = t->value_if<int64_t>(TR_KEY_corrupt_ever).value_or(0); corrupt != 0)
            {
                fmt::print("  Downloaded: {:s} (+{:s} discarded after failed checksum)\n", strlsize(*i), strlsize(corrupt));
            }
            else
            {
                fmt::print("  Downloaded: {:s}\n", strlsize(*i));
            }
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_uploaded_ever))
        {
            fmt::print("  Uploaded: {:s}\n", strlsize(*i));

            if (auto const j = t->value_if<int64_t>(TR_KEY_size_when_done))
            {
                fmt::print("  Ratio: {:s}\n", strlratio(*i, *j));
            }
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_error).value_or(0); i != 0)
        {
            if (auto const sv = t->value_if<std::string_view>(TR_KEY_error_string).value_or(""sv); !std::empty(sv))
            {
                switch (i)
                {
                case TR_STAT_TRACKER_WARNING:
                    fmt::print("  Tracker gave a warning: {:s}\n", sv);
                    break;

                case TR_STAT_TRACKER_ERROR:
                    fmt::print("  Tracker gave an error: {:s}\n", sv);
                    break;

                case TR_STAT_LOCAL_ERROR:
                    fmt::print("  Error: {:s}\n", sv);
                    break;

                default:
                    break; /* no error */
                }
            }
        }

        if (auto i = t->value_if<int64_t>(TR_KEY_peers_connected),
            j = t->value_if<int64_t>(TR_KEY_peers_getting_from_us),
            k = t->value_if<int64_t>(TR_KEY_peers_sending_to_us);
            i && j && k)
        {
            fmt::print("  Peers: connected to {:d}, uploading to {:d}, downloading from {:d}\n", *i, *j, *k);
        }

        if (auto const* const l = t->find_if<tr_variant::Vector>(TR_KEY_webseeds); l != nullptr)
        {
            if (auto const n = std::size(*l); n > 0)
            {
                if (auto const i = t->value_if<int64_t>(TR_KEY_webseeds_sending_to_us))
                {
                    fmt::print("  Web Seeds: downloading from {:d} of {:d} web seeds\n", *i, n);
                }
            }
        }

        fmt::print("\n");

        fmt::print("HISTORY\n");

        if (auto const i = t->value_if<int64_t>(TR_KEY_added_date).value_or(0); i != 0)
        {
            fmt::print("  Date added:       {:s}\n", format_date(buf, i));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_done_date).value_or(0); i != 0)
        {
            fmt::print("  Date finished:    {:s}\n", format_date(buf, i));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_start_date).value_or(0); i != 0)
        {
            fmt::print("  Date started:     {:s}\n", format_date(buf, i));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_activity_date).value_or(0); i != 0)
        {
            fmt::print("  Latest activity:  {:s}\n", format_date(buf, i));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_seconds_downloading).value_or(0); i > 0)
        {
            fmt::print("  Downloading Time: {:s}\n", tr_strltime(i));
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_seconds_seeding).value_or(0); i > 0)
        {
            fmt::print("  Seeding Time:     {:s}\n", tr_strltime(i));
        }

        fmt::print("\n");

        fmt::print("ORIGINS\n");

        if (auto const i = t->value_if<int64_t>(TR_KEY_date_created).value_or(0); i != 0)
        {
            fmt::print("  Date created: {:s}\n", format_date(buf, i));
        }

        if (auto const b = t->value_if<bool>(TR_KEY_is_private))
        {
            fmt::print("  Public torrent: {:s}\n", *b ? "No" : "Yes");
        }

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_comment).value_or(""sv); !std::empty(sv))
        {
            fmt::print("  Comment: {:s}\n", sv);
        }

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_creator).value_or(""sv); !std::empty(sv))
        {
            fmt::print("  Creator: {:s}\n", sv);
        }

        if (auto const sv = t->value_if<std::string_view>(TR_KEY_source).value_or(""sv); !std::empty(sv))
        {
            fmt::print("  Source: {:s}\n", sv);
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_piece_count))
        {
            fmt::print("  Piece Count: {:d}\n", *i);
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_piece_size))
        {
            fmt::print("  Piece Size: {:s}\n", Memory{ *i, Memory::Units::Bytes }.to_string());
        }

        fmt::print("\n");

        fmt::print("LIMITS & BANDWIDTH\n");

        if (auto const b = t->value_if<bool>(TR_KEY_download_limited))
        {
            if (auto const i = t->value_if<int64_t>(TR_KEY_download_limit))
            {
                fmt::print("  Download Limit: ");

                if (*b)
                {
                    fmt::print("{:s}\n", Speed{ *i, Speed::Units::KByps }.to_string());
                }
                else
                {
                    fmt::print("Unlimited\n");
                }
            }
        }

        if (auto b = t->value_if<bool>(TR_KEY_upload_limited))
        {
            if (auto i = t->value_if<int64_t>(TR_KEY_upload_limit))
            {
                fmt::print("  Upload Limit: ");

                if (*b)
                {
                    fmt::print("{:s}\n", Speed{ *i, Speed::Units::KByps }.to_string());
                }
                else
                {
                    fmt::print("Unlimited\n");
                }
            }
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_seed_ratio_mode))
        {
            switch (*i)
            {
            case TR_RATIOLIMIT_GLOBAL:
                fmt::print("  Ratio Limit: Default\n");
                break;

            case TR_RATIOLIMIT_SINGLE:
                if (auto const d = t->value_if<double>(TR_KEY_seed_ratio_limit))
                {
                    fmt::print("  Ratio Limit: {:s}\n", strlratio2(*d));
                }
                break;

            case TR_RATIOLIMIT_UNLIMITED:
                fmt::print("  Ratio Limit: Unlimited\n");
                break;

            default:
                break;
            }
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_seed_idle_mode))
        {
            switch (*i)
            {
            case TR_IDLELIMIT_GLOBAL:
                fmt::print("  Idle Limit: Default\n");
                break;

            case TR_IDLELIMIT_SINGLE:
                if (auto const j = t->value_if<int64_t>(TR_KEY_seed_idle_limit))
                {
                    fmt::print("  Idle Limit: {} minutes\n", *j);
                }
                break;

            case TR_IDLELIMIT_UNLIMITED:
                fmt::print("  Idle Limit: Unlimited\n");
                break;

            default:
                break;
            }
        }

        if (auto const b = t->value_if<bool>(TR_KEY_honors_session_limits))
        {
            fmt::print("  Honors Session Limits: {:s}\n", *b ? "Yes" : "No");
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_peer_limit))
        {
            fmt::print("  Peer limit: {:d}\n", *i);
        }

        if (auto const i = t->value_if<int64_t>(TR_KEY_bandwidth_priority))
        {
            fmt::print("  Bandwidth Priority: {:s}\n", BandwidthPriorityNames[(*i + 1) & 0b11]);
        }

        fmt::print("\n");
    }
}

void print_file_list(tr_variant::Map const& result)
{
    auto* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    for (auto const& t_var : *torrents)
    {
        auto* const t = t_var.get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        auto* const files = t->find_if<tr_variant::Vector>(TR_KEY_files);
        auto* const priorities = t->find_if<tr_variant::Vector>(TR_KEY_priorities);
        auto* const wanteds = t->find_if<tr_variant::Vector>(TR_KEY_wanted);
        auto name = t->value_if<std::string_view>(TR_KEY_name);

        if (!name || files == nullptr || priorities == nullptr || wanteds == nullptr)
        {
            continue;
        }

        auto const n = std::size(*files);
        fmt::print("{:s} ({:d} files):\n", *name, n);
        fmt::print("{:>3s}  {:>5s} {:>8s} {:>3s} {:>9s}  {:s}\n", "#", "Done", "Priority", "Get", "Size", "Name");

        for (size_t i = 0; i < n; ++i)
        {
            auto* const file = (*files)[i].get_if<tr_variant::Map>();
            if (file == nullptr)
            {
                continue;
            }

            auto const have = file->value_if<int64_t>(TR_KEY_bytes_completed);
            auto const length = file->value_if<int64_t>(TR_KEY_length);
            auto const priority = priorities->at(i).value_if<int64_t>();
            auto const wanted = wanteds->at(i).value_if<bool>();
            auto const filename = file->value_if<std::string_view>(TR_KEY_name);

            if (!length || !filename || !have || !priority || !wanted)
            {
                continue;
            }

            static auto constexpr Pristr = [](int64_t p)
            {
                switch (p)
                {
                case TR_PRI_LOW:
                    return "Low"sv;
                case TR_PRI_HIGH:
                    return "High"sv;
                default:
                    return "Normal"sv;
                }
            };

            fmt::print(
                "{:3d}: {:>4s}% {:<8s} {:<3s} {:9s}  {:s}\n",
                i,
                strlpercent(100.0 * *have / *length),
                Pristr(*priority),
                *wanted ? "Yes" : "No",
                strlsize(*length),
                *filename);
        }
    }
}

void print_peers_impl(tr_variant::Vector const& peers)
{
    fmt::print("{:<40s}  {:<12s}  {:<5s} {:<8s}  {:<8s}  {:s}\n", "Address", "Flags", "Done", "Down", "Up", "Client");

    for (auto const& peer_var : peers)
    {
        auto* const peer = peer_var.get_if<tr_variant::Map>();
        if (peer == nullptr)
        {
            continue;
        }

        auto const address = peer->value_if<std::string_view>(TR_KEY_address);
        auto const client = peer->value_if<std::string_view>(TR_KEY_client_name);
        auto const flagstr = peer->value_if<std::string_view>(TR_KEY_flag_str);
        auto const progress = peer->value_if<double>(TR_KEY_progress);
        auto const rate_to_client = peer->value_if<int64_t>(TR_KEY_rate_to_client);
        auto const rate_to_peer = peer->value_if<int64_t>(TR_KEY_rate_to_peer);

        if (address && client && progress && flagstr && rate_to_client && rate_to_peer)
        {
            fmt::print(
                "{:<40s}  {:<12s}  {:<5s} {:8.1f}  {:8.1f}  {:s}\n",
                *address,
                *flagstr,
                strlpercent(*progress * 100.0),
                Speed{ *rate_to_client, Speed::Units::Byps }.count(Speed::Units::KByps),
                Speed{ *rate_to_peer, Speed::Units::Byps }.count(Speed::Units::KByps),
                *client);
        }
    }
}

void print_peers(tr_variant::Map const& result)
{
    auto* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    for (auto it = std::begin(*torrents), end = std::end(*torrents); it != end; ++it)
    {
        auto* const t = it->get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        if (auto* peers = t->find_if<tr_variant::Vector>(TR_KEY_peers); peers != nullptr)
        {
            print_peers_impl(*peers);

            if (it < std::prev(end))
            {
                fmt::print("\n");
            }
        }
    }
}

void print_pieces_impl(std::string_view const raw, size_t const piece_count)
{
    auto const str = tr_base64_decode(raw);
    fmt::print("  ");

    size_t piece = 0;
    static size_t constexpr col_width = 0b111111; // 64 - 1
    for (auto const ch : str)
    {
        for (int bit = 0; piece < piece_count && bit < 8; ++bit, ++piece)
        {
            fmt::print("{:c}", (ch & (1 << (7 - bit))) != 0 ? '1' : '0');
        }

        fmt::print(" ");

        if ((piece & col_width) == 0) // piece % 64 == 0
        {
            fmt::print("\n  ");
        }
    }

    fmt::print("\n");
}

void print_pieces(tr_variant::Map const& result)
{
    auto const* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    for (auto it = std::cbegin(*torrents), end = std::cend(*torrents); it != end; ++it)
    {
        auto const* const t = it->get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        auto const piece_count = t->value_if<int64_t>(TR_KEY_piece_count);
        auto const pieces = t->value_if<std::string_view>(TR_KEY_pieces);

        if (!piece_count || !pieces)
        {
            continue;
        }

        TR_ASSERT(*piece_count >= 0);
        print_pieces_impl(*pieces, static_cast<size_t>(*piece_count));

        if (it < std::prev(end))
        {
            fmt::print("\n");
        }
    }
}

void print_port_test(tr_variant::Map const& result)
{
    if (auto is_open = result.value_if<bool>(TR_KEY_port_is_open); is_open)
    {
        fmt::print("Port is open: {:s}\n", *is_open ? "Yes" : "No");
    }
}

void print_torrent_list(tr_variant::Map const& result)
{
    auto const* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    fmt::print(
        "{:>6s}   {:>5s}  {:>9s}  {:<9s}  {:>8s}  {:>8s}  {:<5s}  {:<11s}  {:<s}\n",
        "ID",
        "Done",
        "Have",
        "ETA",
        "Up",
        "Down",
        "Ratio",
        "Status",
        "Name");

    auto tptrs = std::vector<tr_variant::Map const*>{};
    tptrs.reserve(std::size(*torrents));
    for (auto const& t_var : *torrents)
    {
        if (auto* t = t_var.get_if<tr_variant::Map>(); t != nullptr && t->value_if<int64_t>(TR_KEY_id))
        {
            tptrs.push_back(t);
        }
    }

    std::sort(
        tptrs.begin(),
        tptrs.end(),
        [](tr_variant::Map const* f, tr_variant::Map const* s)
        {
            static auto constexpr Min = std::numeric_limits<int64_t>::min();
            auto const f_time = f->value_if<int64_t>(TR_KEY_added_date).value_or(Min);
            auto const s_time = s->value_if<int64_t>(TR_KEY_added_date).value_or(Min);
            return f_time < s_time;
        });

    int64_t total_size = 0;
    int64_t total_up = 0;
    int64_t total_down = 0;
    for (auto const& t : tptrs)
    {
        auto o_tor_id = t->value_if<int64_t>(TR_KEY_id);
        auto o_eta = t->value_if<int64_t>(TR_KEY_eta);
        auto o_status = t->value_if<int64_t>(TR_KEY_status);
        auto o_up = t->value_if<int64_t>(TR_KEY_rate_upload);
        auto o_down = t->value_if<int64_t>(TR_KEY_rate_download);
        auto o_size_when_done = t->value_if<int64_t>(TR_KEY_size_when_done);
        auto o_left_until_done = t->value_if<int64_t>(TR_KEY_left_until_done);
        auto o_ratio = t->value_if<double>(TR_KEY_upload_ratio);
        auto o_name = t->value_if<std::string_view>(TR_KEY_name);

        if (!o_eta || !o_tor_id || !o_left_until_done || !o_name || !o_down || !o_up || !o_size_when_done || !o_status ||
            !o_ratio)
        {
            continue;
        }

        auto const eta = *o_eta;
        auto const up = *o_up;
        auto const down = *o_down;
        auto const size_when_done = *o_size_when_done;
        auto const left_until_done = *o_left_until_done;

        auto const eta_str = left_until_done != 0 || eta != -1 ? eta_to_string(eta) : "Done";
        auto const error_mark = t->value_if<int64_t>(TR_KEY_error).value_or(0) != 0 ? '*' : ' ';
        auto const done_str = size_when_done != 0 ?
            strlpercent(100.0 * (size_when_done - left_until_done) / size_when_done) + '%' :
            std::string{ "n/a" };

        fmt::print(
            "{:>6d}{:c}  {:>5s}  {:>9s}  {:<9s}  {:8.1f}  {:8.1f}  {:>5s}  {:<11s}  {:<s}\n",
            *o_tor_id,
            error_mark,
            done_str,
            strlsize(size_when_done - left_until_done),
            eta_str,
            Speed{ up, Speed::Units::Byps }.count(Speed::Units::KByps),
            Speed{ down, Speed::Units::Byps }.count(Speed::Units::KByps),
            strlratio2(*o_ratio),
            get_status_string(*t),
            *o_name);

        total_up += up;
        total_down += down;
        total_size += size_when_done - left_until_done;
    }

    fmt::print(
        "Sum:            {:>9s}             {:8.1f}  {:8.1f}\n",
        strlsize(total_size).c_str(),
        Speed{ total_up, Speed::Units::Byps }.count(Speed::Units::KByps),
        Speed{ total_down, Speed::Units::Byps }.count(Speed::Units::KByps));
}

void print_trackers_impl(tr_variant::Vector const& tracker_stats)
{
    for (auto const& t_var : tracker_stats)
    {
        auto* const t = t_var.get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        auto const announce_state = t->value_if<int64_t>(TR_KEY_announce_state);
        auto const download_count = t->value_if<int64_t>(TR_KEY_download_count);
        auto const has_announced = t->value_if<bool>(TR_KEY_has_announced);
        auto const has_scraped = t->value_if<bool>(TR_KEY_has_scraped);
        auto const host = t->value_if<std::string_view>(TR_KEY_host);
        auto const is_backup = t->value_if<bool>(TR_KEY_is_backup);
        auto const last_announce_peer_count = t->value_if<int64_t>(TR_KEY_last_announce_peer_count);
        auto const last_announce_result = t->value_if<std::string_view>(TR_KEY_last_announce_result);
        auto const last_announce_start_time = t->value_if<int64_t>(TR_KEY_last_announce_start_time);
        auto const last_announce_time = t->value_if<int64_t>(TR_KEY_last_announce_time);
        auto const last_scrape_result = t->value_if<std::string_view>(TR_KEY_last_scrape_result);
        auto const last_scrape_start_time = t->value_if<int64_t>(TR_KEY_last_scrape_start_time);
        auto const last_scrape_succeeded = t->value_if<bool>(TR_KEY_last_scrape_succeeded);
        auto const last_scrape_time = t->value_if<int64_t>(TR_KEY_last_scrape_time);
        auto const last_scrape_timed_out = t->value_if<bool>(TR_KEY_last_scrape_timed_out);
        auto const leecher_count = t->value_if<int64_t>(TR_KEY_leecher_count);
        auto const next_announce_time = t->value_if<int64_t>(TR_KEY_next_announce_time);
        auto const next_scrape_time = t->value_if<int64_t>(TR_KEY_next_scrape_time);
        auto const scrape_state = t->value_if<int64_t>(TR_KEY_scrape_state);
        auto const seeder_count = t->value_if<int64_t>(TR_KEY_seeder_count);
        auto const tier = t->value_if<int64_t>(TR_KEY_tier);
        auto const tracker_id = t->value_if<int64_t>(TR_KEY_id);
        auto const last_announce_succeeded = t->value_if<bool>(TR_KEY_last_announce_succeeded);
        auto const last_announce_timed_out = t->value_if<bool>(TR_KEY_last_announce_timed_out);

        if (!download_count || !has_announced || !has_scraped || !host || !tracker_id || !is_backup || !announce_state ||
            !scrape_state || !last_announce_peer_count || !last_announce_result || !last_announce_start_time ||
            !last_announce_succeeded || !last_announce_time || !last_announce_timed_out || !last_scrape_result ||
            !last_scrape_start_time || !last_scrape_succeeded || !last_scrape_time || !last_scrape_timed_out ||
            !leecher_count || !next_announce_time || !next_scrape_time || !seeder_count || !tier)
        {
            continue;
        }

        time_t const now = time(nullptr);

        fmt::print("\n");
        fmt::print("  Tracker {:d}: {:s}\n", *tracker_id, *host);

        if (*is_backup)
        {
            fmt::print("  Backup on tier {:d}\n", *tier);
            continue;
        }
        fmt::print("  Active in tier {:d}\n", *tier);

        if (*has_announced && *announce_state != TR_TRACKER_INACTIVE)
        {
            auto const timestr = tr_strltime(now - *last_announce_time);

            if (*last_announce_succeeded)
            {
                fmt::print("  Got a list of {:d} peers {:s} ago\n", *last_announce_peer_count, timestr);
            }
            else if (*last_announce_timed_out)
            {
                fmt::print("  Peer list request timed out; will retry\n");
            }
            else
            {
                fmt::print("  Got an error '{:s}' {:s} ago\n", *last_announce_result, timestr);
            }
        }

        switch (*announce_state)
        {
        case TR_TRACKER_INACTIVE:
            fmt::print("  No updates scheduled\n");
            break;

        case TR_TRACKER_WAITING:
            fmt::print("  Asking for more peers in {:s}\n", tr_strltime(*next_announce_time - now));
            break;

        case TR_TRACKER_QUEUED:
            fmt::print("  Queued to ask for more peers\n");
            break;

        case TR_TRACKER_ACTIVE:
            fmt::print("  Asking for more peers now... {:s}\n", tr_strltime(now - *last_announce_start_time));
            break;

        default:
            break;
        }

        if (*has_scraped)
        {
            auto const timestr = tr_strltime(now - *last_scrape_time);

            if (*last_scrape_succeeded)
            {
                fmt::print("  Tracker had {:d} seeders and {:d} leechers {:s} ago\n", *seeder_count, *leecher_count, timestr);
            }
            else if (*last_scrape_timed_out)
            {
                fmt::print("  Tracker scrape timed out; will retry\n");
            }
            else
            {
                fmt::print("  Got a scrape error '{:s}' {:s} ago\n", *last_scrape_result, timestr);
            }
        }

        switch (*scrape_state)
        {
        case TR_TRACKER_WAITING:
            fmt::print("  Asking for peer counts in {:s}\n", tr_strltime(*next_scrape_time - now));
            break;

        case TR_TRACKER_QUEUED:
            fmt::print("  Queued to ask for peer counts\n");
            break;

        case TR_TRACKER_ACTIVE:
            fmt::print("  Asking for peer counts now... {:s}\n", tr_strltime(now - *last_scrape_start_time));
            break;

        default: // TR_TRACKER_INACTIVE
            break;
        }
    }
}

void print_trackers(tr_variant::Map const& result)
{
    auto const* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    for (auto it = std::cbegin(*torrents), end = std::cend(*torrents); it != end; ++it)
    {
        auto const* const t = it->get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        if (auto const* const tracker_stats = t->find_if<tr_variant::Vector>(TR_KEY_tracker_stats))
        {
            print_trackers_impl(*tracker_stats);

            if (it < std::prev(end))
            {
                fmt::print("\n");
            }
        }
    }
}

void print_session(tr_variant::Map const& result)
{
    fmt::print("VERSION\n");

    if (auto sv = result.value_if<std::string_view>(TR_KEY_version))
    {
        fmt::print("  Daemon version: {:s}\n", *sv);
    }

    if (auto i = result.value_if<int64_t>(TR_KEY_rpc_version))
    {
        fmt::print("  RPC version: {:d}\n", *i);
    }

    if (auto i = result.value_if<int64_t>(TR_KEY_rpc_version_minimum))
    {
        fmt::print("  RPC minimum version: {:d}\n", *i);
    }

    fmt::print("\n");

    fmt::print("CONFIG\n");

    if (auto sv = result.value_if<std::string_view>(TR_KEY_config_dir))
    {
        fmt::print("  Configuration directory: {:s}\n", *sv);
    }

    if (auto sv = result.value_if<std::string_view>(TR_KEY_download_dir))
    {
        fmt::print("  Download directory: {:s}\n", *sv);
    }

    if (auto i = result.value_if<int64_t>(TR_KEY_peer_port))
    {
        fmt::print("  Listen port: {:d}\n", *i);
    }

    if (auto b = result.value_if<bool>(TR_KEY_port_forwarding_enabled))
    {
        fmt::print("  Port forwarding enabled: {:s}\n", *b ? "Yes" : "No");
    }

    if (auto b = result.value_if<bool>(TR_KEY_utp_enabled))
    {
        fmt::print("  µTP enabled: {:s}\n", *b ? "Yes" : "No");
    }

    if (auto b = result.value_if<bool>(TR_KEY_dht_enabled))
    {
        fmt::print("  Distributed hash table enabled: {:s}\n", *b ? "Yes" : "No");
    }

    if (auto b = result.value_if<bool>(TR_KEY_lpd_enabled))
    {
        fmt::print("  Local peer discovery enabled: {:s}\n", *b ? "Yes" : "No");
    }

    if (auto b = result.value_if<bool>(TR_KEY_pex_enabled))
    {
        fmt::print("  Peer exchange allowed: {:s}\n", *b ? "Yes" : "No");
    }

    if (auto sv = result.value_if<std::string_view>(TR_KEY_encryption))
    {
        fmt::print("  Encryption: {:s}\n", *sv);
    }

    if (auto i = result.value_if<int64_t>(TR_KEY_cache_size_mib))
    {
        fmt::print("  Maximum memory cache size: {:s}\n", Memory{ *i, Memory::Units::MBytes }.to_string());
    }

    if (auto b = result.value_if<bool>(TR_KEY_sequential_download))
    {
        fmt::print("  Sequential download: {:s}\n", *b ? "Yes" : "No");
    }

    auto const alt_enabled = result.value_if<bool>(TR_KEY_alt_speed_enabled);
    auto const alt_time_enabled = result.value_if<bool>(TR_KEY_alt_speed_time_enabled);
    auto const up_enabled = result.value_if<bool>(TR_KEY_speed_limit_up_enabled);
    auto const down_enabled = result.value_if<bool>(TR_KEY_speed_limit_down_enabled);
    auto const speed_ratio_limited = result.value_if<bool>(TR_KEY_seed_ratio_limited);
    auto const idle_seeding_limited = result.value_if<bool>(TR_KEY_idle_seeding_limit_enabled);
    auto const alt_down = result.value_if<int64_t>(TR_KEY_alt_speed_down);
    auto const alt_up = result.value_if<int64_t>(TR_KEY_alt_speed_up);
    auto const alt_begin = result.value_if<int64_t>(TR_KEY_alt_speed_time_begin);
    auto const alt_end = result.value_if<int64_t>(TR_KEY_alt_speed_time_end);
    auto const alt_day = result.value_if<int64_t>(TR_KEY_alt_speed_time_day);
    auto const up_limit = result.value_if<int64_t>(TR_KEY_speed_limit_up);
    auto const down_limit = result.value_if<int64_t>(TR_KEY_speed_limit_down);
    auto const peer_limit = result.value_if<int64_t>(TR_KEY_peer_limit_global);
    auto const idle_seeding_limit = result.value_if<int64_t>(TR_KEY_idle_seeding_limit);
    auto const seed_ratio_limit = result.value_if<double>(TR_KEY_seed_ratio_limit);

    if (alt_down && alt_enabled && alt_begin && alt_time_enabled && alt_end && alt_day && alt_up && peer_limit && down_limit &&
        down_enabled && up_limit && up_enabled && seed_ratio_limit && speed_ratio_limited && idle_seeding_limited &&
        idle_seeding_limit)
    {
        fmt::print("\n");
        fmt::print("LIMITS\n");
        fmt::print("  Peer limit: {:d}\n", *peer_limit);

        fmt::print("  Default seed ratio limit: {:s}\n", *speed_ratio_limited ? strlratio2(*seed_ratio_limit) : "Unlimited");

        fmt::print(
            "  Default idle seeding time limit: {:s}\n",
            *idle_seeding_limited ? std::to_string(*idle_seeding_limit) + " minutes" : "Unlimited");

        std::string effective_up_limit;

        if (*alt_enabled)
        {
            effective_up_limit = Speed{ *alt_up, Speed::Units::KByps }.to_string();
        }
        else if (*up_enabled)
        {
            effective_up_limit = Speed{ *up_limit, Speed::Units::KByps }.to_string();
        }
        else
        {
            effective_up_limit = "Unlimited"s;
        }

        fmt::print(
            "  Upload speed limit: {:s} ({:s} limit: {:s}; {:s} turtle limit: {:s})\n",
            effective_up_limit,
            *up_enabled ? "Enabled" : "Disabled",
            Speed{ *up_limit, Speed::Units::KByps }.to_string(),
            *alt_enabled ? "Enabled" : "Disabled",
            Speed{ *alt_up, Speed::Units::KByps }.to_string());

        std::string effective_down_limit;

        if (*alt_enabled)
        {
            effective_down_limit = Speed{ *alt_down, Speed::Units::KByps }.to_string();
        }
        else if (*down_enabled)
        {
            effective_down_limit = Speed{ *down_limit, Speed::Units::KByps }.to_string();
        }
        else
        {
            effective_down_limit = "Unlimited"s;
        }

        fmt::print(
            "  Download speed limit: {:s} ({:s} limit: {:s}; {:s} turtle limit: {:s})\n",
            effective_down_limit,
            *down_enabled ? "Enabled" : "Disabled",
            Speed{ *down_limit, Speed::Units::KByps }.to_string(),
            *alt_enabled ? "Enabled" : "Disabled",
            Speed{ *alt_down, Speed::Units::KByps }.to_string());

        if (*alt_time_enabled)
        {
            fmt::print(
                "  Turtle schedule: {:02d}:{:02d} - {:02d}:{:02d}  ",
                *alt_begin / 60,
                *alt_begin % 60,
                *alt_end / 60,
                *alt_end % 60);

            if ((*alt_day & TR_SCHED_SUN) != 0)
            {
                fmt::print("Sun ");
            }

            if ((*alt_day & TR_SCHED_MON) != 0)
            {
                fmt::print("Mon ");
            }

            if ((*alt_day & TR_SCHED_TUES) != 0)
            {
                fmt::print("Tue ");
            }

            if ((*alt_day & TR_SCHED_WED) != 0)
            {
                fmt::print("Wed ");
            }

            if ((*alt_day & TR_SCHED_THURS) != 0)
            {
                fmt::print("Thu ");
            }

            if ((*alt_day & TR_SCHED_FRI) != 0)
            {
                fmt::print("Fri ");
            }

            if ((*alt_day & TR_SCHED_SAT) != 0)
            {
                fmt::print("Sat ");
            }

            fmt::print("\n");
        }
    }

    fmt::print("\n");

    fmt::print("MISC\n");

    if (auto b = result.value_if<bool>(TR_KEY_start_added_torrents))
    {
        fmt::print("  Autostart added torrents: {:s}\n", *b ? "Yes" : "No");
    }

    if (auto b = result.value_if<bool>(TR_KEY_trash_original_torrent_files))
    {
        fmt::print("  Delete automatically added torrents: {:s}\n", *b ? "Yes" : "No");
    }
}

void print_session_stats(tr_variant::Map const& result)
{
    if (auto const* const stats = result.find_if<tr_variant::Map>(TR_KEY_current_stats))
    {
        auto const up = stats->value_if<int64_t>(TR_KEY_uploaded_bytes);
        auto const down = stats->value_if<int64_t>(TR_KEY_downloaded_bytes);
        auto const secs = stats->value_if<int64_t>(TR_KEY_seconds_active);

        if (up && down && secs)
        {
            fmt::print("\nCURRENT SESSION\n");
            fmt::print("  Uploaded:   {:s}\n", strlsize(*up));
            fmt::print("  Downloaded: {:s}\n", strlsize(*down));
            fmt::print("  Ratio:      {:s}\n", strlratio(*up, *down));
            fmt::print("  Duration:   {:s}\n", tr_strltime(*secs));
        }
    }

    if (auto const* const stats = result.find_if<tr_variant::Map>(TR_KEY_cumulative_stats))
    {
        auto const up = stats->value_if<int64_t>(TR_KEY_uploaded_bytes);
        auto const down = stats->value_if<int64_t>(TR_KEY_downloaded_bytes);
        auto const secs = stats->value_if<int64_t>(TR_KEY_seconds_active);
        auto const sessions = stats->value_if<int64_t>(TR_KEY_session_count);

        if (up && down && secs && sessions)
        {
            fmt::print("\nTOTAL\n");
            fmt::print("  Started {:d} times\n", *sessions);
            fmt::print("  Uploaded:   {:s}\n", strlsize(*up));
            fmt::print("  Downloaded: {:s}\n", strlsize(*down));
            fmt::print("  Ratio:      {:s}\n", strlratio(*up, *down));
            fmt::print("  Duration:   {:s}\n", tr_strltime(*secs));
        }
    }
}

void print_groups(tr_variant::Map const& result)
{
    auto const* const groups = result.find_if<tr_variant::Vector>(TR_KEY_group);
    if (groups == nullptr)
    {
        return;
    }

    for (auto const& group_var : *groups)
    {
        auto const* const group = group_var.get_if<tr_variant::Map>();
        if (group == nullptr)
        {
            continue;
        }

        auto const name = group->value_if<std::string_view>(TR_KEY_name);
        auto const up_enabled = group->value_if<bool>(TR_KEY_upload_limited);
        auto const down_enabled = group->value_if<bool>(TR_KEY_download_limited);
        auto const up_limit = group->value_if<int64_t>(TR_KEY_upload_limit);
        auto const down_limit = group->value_if<int64_t>(TR_KEY_download_limit);
        auto const honors = group->value_if<bool>(TR_KEY_honors_session_limits);
        if (name && down_limit && down_enabled && up_limit && up_enabled && honors)
        {
            fmt::print("{:s}: ", *name);
            fmt::print(
                "Upload speed limit: {:s}, Download speed limit: {:s}, {:s} session bandwidth limits\n",
                *up_enabled ? Speed{ *up_limit, Speed::Units::KByps }.to_string() : "unlimited"s,
                *down_enabled ? Speed{ *down_limit, Speed::Units::KByps }.to_string() : "unlimited"s,
                *honors ? "honors" : "does not honor");
        }
    }
}

void filter_ids(tr_variant::Map const& result, RemoteConfig& config)
{
    auto const* const torrents = result.find_if<tr_variant::Vector>(TR_KEY_torrents);
    if (torrents == nullptr)
    {
        return;
    }

    std::set<int64_t> ids;

    size_t pos = 0;
    bool negate = false;
    std::string_view arg;

    if (config.filter[pos] == '~')
    {
        ++pos;
        negate = true;
    }
    if (std::size(config.filter) > pos + 1 && config.filter[pos + 1] == ':')
    {
        arg = &config.filter[pos + 2];
    }

    for (auto const& t_var : *torrents)
    {
        auto const* const t = t_var.get_if<tr_variant::Map>();
        if (t == nullptr)
        {
            continue;
        }

        auto const tor_id = t->value_if<int64_t>(TR_KEY_id).value_or(-1);
        if (tor_id < 0)
        {
            continue;
        }

        bool include = negate;
        auto const status = get_status_string(*t);
        switch (config.filter[pos])
        {
        case 'i': // Status = Idle
            if (status == "Idle")
            {
                include = !include;
            }
            break;
        case 'd': // Downloading (Status is Downloading or Up&Down)
            if (status.find("Down") != std::string::npos)
            {
                include = !include;
            }
            break;
        case 'u': // Uploading (Status is Uploading, Up&Down or Seeding
            if (status.find("Up") != std::string::npos || status == "Seeding")
            {
                include = !include;
            }
            break;
        case 'l': // label
            if (auto* l = t->find_if<tr_variant::Vector>(TR_KEY_labels); l != nullptr)
            {
                for (auto const& label_var : *l)
                {
                    if (auto sv = label_var.value_if<std::string_view>(); sv && arg == *sv)
                    {
                        include = !include;
                        break;
                    }
                }
            }
            break;
        case 'n': // Torrent name substring
            if (auto name = t->value_if<std::string_view>(TR_KEY_name); !name)
            {
                continue;
            }
            else if (name->find(arg) != std::string::npos)
            {
                include = !include;
            }
            break;
        case 'r': // Minimal ratio
            if (auto ratio = t->value_if<double>(TR_KEY_upload_ratio); !ratio)
            {
                continue;
            }
            else if (*ratio >= std::stof(std::string{ arg }))
            {
                include = !include;
            }
            break;
        case 'w': // Not all torrent wanted
            if (auto total_size = t->value_if<int64_t>(TR_KEY_total_size).value_or(-1); total_size < 0)
            {
                continue;
            }
            else if (auto size_when_done = t->value_if<int64_t>(TR_KEY_size_when_done).value_or(-1); size_when_done < 0)
            {
                continue;
            }
            else if (total_size > size_when_done)
            {
                include = !include;
            }
            break;
        default:
            break;
        }

        if (include)
        {
            ids.insert(tor_id);
        }
    }

    auto& res = config.torrent_ids;
    res.clear();
    for (auto const& i : ids)
    {
        res += std::to_string(i) + ",";
    }
    if (res.empty())
    {
        res = ","; // no selected torrents
    }
}

void print_blocklist_size(tr_variant::Map const& result)
{
    if (auto const i = result.value_if<int64_t>(TR_KEY_blocklist_size))
    {
        fmt::print("Blocklist size: {:d}\n", *i);
    }
}

int process_response(char const* rpcurl, std::string_view const response, RemoteConfig& config)
{
    if (config.json)
    {
        fmt::print("{:s}\n", response);
        return EXIT_SUCCESS;
    }

    if (config.debug)
    {
        fmt::print(stderr, "got response (len {:d}):\n--------\n{:s}\n--------\n", std::size(response), response);
    }

    auto parsed = tr_variant_serde::json().inplace().parse(response);
    if (!parsed)
    {
        fmt::print(stderr, "Unable to parse response '{}'\n", response);
        return EXIT_FAILURE;
    }
    api_compat::convert_incoming_data(*parsed);

    auto const* const top = parsed->get_if<tr_variant::Map>();
    if (!top)
    {
        fmt::print(stderr, "Response was not a JSON object\n");
        return EXIT_FAILURE;
    }

    if (auto const jsonrpc = top->value_if<std::string_view>(TR_KEY_jsonrpc); jsonrpc != JsonRpc::Version)
    {
        fmt::print(stderr, "Response was not JSON-RPC {:s}\n", JsonRpc::Version);
        return EXIT_FAILURE;
    }

    if (auto const* const errmap = top->find_if<tr_variant::Map>(TR_KEY_error))
    {
        auto errmsg = std::optional<std::string_view>{};

        if (auto const* const data = errmap->find_if<tr_variant::Map>(TR_KEY_data))
        {
            errmsg = data->value_if<std::string_view>(TR_KEY_error_string);
        }

        if (!errmsg)
        {
            errmsg = errmap->value_if<std::string_view>(TR_KEY_message);
        }

        fmt::print("Error: {:s}\n", errmsg.value_or("unknown error"sv));
        return EXIT_FAILURE;
    }

    auto empty_result = tr_variant::Map{};
    auto* result = top->find_if<tr_variant::Map>(TR_KEY_result);
    result = result ? result : &empty_result;

    switch (top->value_if<int64_t>(TR_KEY_id).value_or(ID_NOOP))
    {
    case ID_SESSION:
        print_session(*result);
        break;

    case ID_STATS:
        print_session_stats(*result);
        break;

    case ID_DETAILS:
        print_details(*result);
        break;

    case ID_FILES:
        print_file_list(*result);
        break;

    case ID_PEERS:
        print_peers(*result);
        break;

    case ID_LIST:
        print_torrent_list(*result);
        break;

    case ID_TRACKERS:
        print_trackers(*result);
        break;

    case ID_PORTTEST:
        print_port_test(*result);
        break;

    case ID_PIECES:
        print_pieces(*result);
        break;

    case ID_GROUPS:
        print_groups(*result);
        break;

    case ID_FILTER:
        filter_ids(*result, config);
        break;

    case ID_BLOCKLIST:
        print_blocklist_size(*result);
        break;

    case ID_TORRENT_ADD:
        if (auto* b = result->find_if<tr_variant::Map>(TR_KEY_torrent_added))
        {
            if (auto const id = b->value_if<int64_t>(TR_KEY_id))
            {
                config.torrent_ids = std::to_string(*id);
            }
        }
        [[fallthrough]];

    default:
        fmt::print("{:s} responded: {:s}\n", rpcurl, response);
        break;
    }

    return EXIT_SUCCESS;
}

namespace flush_utils
{
CURL* tr_curl_easy_init(std::string* writebuf, RemoteConfig& config)
{
    CURL* curl = curl_easy_init();
    (void)curl_easy_setopt(curl, CURLOPT_USERAGENT, fmt::format("{:s}/{:s}", MyName, LONG_VERSION_STRING).c_str());
    (void)curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, write_func);
    (void)curl_easy_setopt(curl, CURLOPT_WRITEDATA, writebuf);
    (void)curl_easy_setopt(curl, CURLOPT_HEADERDATA, &config);
    (void)curl_easy_setopt(curl, CURLOPT_HEADERFUNCTION, parse_response_header);
    (void)curl_easy_setopt(curl, CURLOPT_POST, 1);
    (void)curl_easy_setopt(curl, CURLOPT_NETRC, CURL_NETRC_OPTIONAL);
    (void)curl_easy_setopt(curl, CURLOPT_HTTPAUTH, CURLAUTH_ANY);
    (void)curl_easy_setopt(curl, CURLOPT_VERBOSE, config.debug);
    (void)curl_easy_setopt(
        curl,
        CURLOPT_ENCODING,
        ""); /* "" tells curl to fill in the blanks with what it was compiled to support */

    if (auto const& str = config.unix_socket_path; !std::empty(str))
    {
        (void)curl_easy_setopt(curl, CURLOPT_UNIX_SOCKET_PATH, str.c_str());
    }

    if (auto const& str = config.netrc; !std::empty(str))
    {
        (void)curl_easy_setopt(curl, CURLOPT_NETRC_FILE, str.c_str());
    }

    if (auto const& str = config.auth; !std::empty(str))
    {
        (void)curl_easy_setopt(curl, CURLOPT_USERPWD, str.c_str());
    }

    if (config.use_ssl)
    {
        // do not verify subject/hostname
        (void)curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0);

        // since most certs will be self-signed, do not verify against CA
        (void)curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0);
    }

    if (auto const& str = config.session_id; !std::empty(str))
    {
        auto const h = fmt::format("{:s}: {:s}", TR_RPC_SESSION_ID_HEADER, str);
        auto* const custom_headers = curl_slist_append(nullptr, h.c_str());

        (void)curl_easy_setopt(curl, CURLOPT_HTTPHEADER, custom_headers);
        (void)curl_easy_setopt(curl, CURLOPT_PRIVATE, custom_headers);
    }

    return curl;
}

void tr_curl_easy_cleanup(CURL* curl)
{
    struct curl_slist* custom_headers = nullptr;
    curl_easy_getinfo(curl, CURLINFO_PRIVATE, &custom_headers);

    curl_easy_cleanup(curl);

    if (custom_headers != nullptr)
    {
        curl_slist_free_all(custom_headers);
    }
}

[[nodiscard]] long get_timeout_secs(std::string_view req)
{
    if (tr_strv_contains(req, R"("method":"blocklist-update")") || tr_strv_contains(req, R"("method":"blocklist_update")"))
    {
        return 300L;
    }

    return 60L; /* default value */
}
} // namespace flush_utils

int flush(char const* rpcurl, tr_variant* const var, RemoteConfig& config)
{
    using namespace flush_utils;

    api_compat::convert(*var, config.network_style);
    auto const payload = tr_variant_serde::json().compact().to_string(*var);
    auto const scheme = config.use_ssl ? "https"sv : "http"sv;
    auto const rpcurl_http = fmt::format("{:s}://{:s}", scheme, rpcurl);

    auto buf = std::string{};
    auto* curl = tr_curl_easy_init(&buf, config);
    (void)curl_easy_setopt(curl, CURLOPT_URL, rpcurl_http.c_str());
    (void)curl_easy_setopt(curl, CURLOPT_POSTFIELDS, payload.c_str());
    (void)curl_easy_setopt(curl, CURLOPT_TIMEOUT, get_timeout_secs(payload));

    if (config.debug)
    {
        fmt::print(stderr, "posting:\n--------\n{:s}\n--------\n", payload);
    }

    auto status = EXIT_SUCCESS;
    if (auto const res = curl_easy_perform(curl); res != CURLE_OK)
    {
        fmt::print(stderr, "Unable to send request to '{}': {}\n", rpcurl_http, curl_easy_strerror(res));
        status |= EXIT_FAILURE;
    }
    else
    {
        long response;
        curl_easy_getinfo(curl, CURLINFO_RESPONSE_CODE, &response);

        switch (response)
        {
        case 200:
            status |= process_response(rpcurl, buf, config);
            break;

        case 204:
            fmt::print("{:s} acknowledged request\n", rpcurl);
            break;

        case 409:
            // Session id failed. Our curl header func has already
            // pulled the new session id from this response's headers.
            // Build a new CURL* and try again
            tr_curl_easy_cleanup(curl);
            curl = nullptr;
            status |= flush(rpcurl, var, config);
            break;

        default:
            fmt::print(stderr, "Unexpected response: {:s}\n", buf);
            status |= EXIT_FAILURE;
            break;
        }
    }

    /* cleanup */
    if (curl != nullptr)
    {
        tr_curl_easy_cleanup(curl);
    }

    var->clear();

    return status;
}

tr_variant::Map& ensure_sset(tr_variant& sset)
{
    auto* map = sset.get_if<tr_variant::Map>();
    if (map == nullptr)
    {
        sset = tr_variant::Map{ 4U };
        map = sset.get_if<tr_variant::Map>();
        map->try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
        map->try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_session_set));
        map->try_emplace(TR_KEY_id, ID_NOOP);
    }

    auto* params = map->find_if<tr_variant::Map>(TR_KEY_params);
    if (params == nullptr)
    {
        params = map->insert_or_assign(TR_KEY_params, tr_variant::Map{}).first.get_if<tr_variant::Map>();
    }
    return *params;
}

tr_variant::Map& ensure_tset(tr_variant& tset)
{
    auto* map = tset.get_if<tr_variant::Map>();
    if (map == nullptr)
    {
        tset = tr_variant::Map{ 4U };
        map = tset.get_if<tr_variant::Map>();
        map->try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
        map->try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_set));
        map->try_emplace(TR_KEY_id, ID_NOOP);
    }

    auto* params = map->find_if<tr_variant::Map>(TR_KEY_params);
    if (params == nullptr)
    {
        params = map->insert_or_assign(TR_KEY_params, tr_variant::Map{ 1 }).first.get_if<tr_variant::Map>();
    }
    return *params;
}

tr_variant::Map& ensure_tadd(tr_variant& tadd)
{
    auto* map = tadd.get_if<tr_variant::Map>();
    if (map == nullptr)
    {
        tadd = tr_variant::Map{ 4U };
        map = tadd.get_if<tr_variant::Map>();
        map->try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
        map->try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_add));
        map->try_emplace(TR_KEY_id, ID_TORRENT_ADD);
    }

    auto* args = map->find_if<tr_variant::Map>(TR_KEY_params);
    if (args == nullptr)
    {
        args = map->insert_or_assign(TR_KEY_params, tr_variant::Map{}).first.get_if<tr_variant::Map>();
    }
    return *args;
}

int process_args(char const* rpcurl, int argc, char const* const* argv, RemoteConfig& config)
{
    auto status = int{ EXIT_SUCCESS };
    char const* optarg;
    auto sset = tr_variant{};
    auto tset = tr_variant{};
    auto tadd = tr_variant{};
    auto rename_from = std::string{};

    for (;;)
    {
        int const c = tr_getopt(Usage, argc, argv, std::data(Options), &optarg);
        if (c == TR_OPT_DONE)
        {
            break;
        }

        auto const optarg_sv = std::string_view{ optarg != nullptr ? optarg : "" };
        if (auto const step_mode = get_opt_mode(c); step_mode == MODE_META_COMMAND) /* meta commands */
        {
            switch (c)
            {
            case 'a': /* add torrent */
                if (sset.has_value())
                {
                    status |= flush(rpcurl, &sset, config);
                }

                if (tadd.has_value())
                {
                    status |= flush(rpcurl, &tadd, config);
                }

                if (auto* tset_map = tset.get_if<tr_variant::Map>(); tset_map != nullptr)
                {
                    auto* const params = tset_map->find_if<tr_variant::Map>(TR_KEY_params);
                    TR_ASSERT(params != nullptr);
                    if (params != nullptr)
                    {
                        add_id_arg(*params, config);
                        status |= flush(rpcurl, &tset, config);
                    }
                }

                ensure_tadd(tadd);
                break;

            case 'b': /* debug */
                config.debug = true;
                break;

            case 'j': /* return output as JSON */
                config.json = true;
                break;

            case 968: /* Unix domain socket */
                config.unix_socket_path = optarg_sv;
                break;

            case 'n': /* auth */
                config.auth = optarg_sv;
                break;

            case 810: /* authenv */
                if (auto const authstr = tr_env_get_string("TR_AUTH"); !std::empty(authstr))
                {
                    config.auth = authstr;
                }
                else
                {
                    fmt::print(stderr, "The TR_AUTH environment variable is not set\n");
                    exit(0);
                }

                break;

            case 'N':
                config.netrc = optarg_sv;
                break;

            case 820:
                config.use_ssl = true;
                break;

            case 't': /* set current torrent */
                if (tadd.has_value())
                {
                    status |= flush(rpcurl, &tadd, config);
                }

                if (auto* tset_map = tset.get_if<tr_variant::Map>(); tset_map != nullptr)
                {
                    auto* const params = tset_map->find_if<tr_variant::Map>(TR_KEY_params);
                    TR_ASSERT(params != nullptr);
                    if (params != nullptr)
                    {
                        add_id_arg(*params, config);
                        status |= flush(rpcurl, &tset, config);
                    }
                }

                config.torrent_ids = optarg_sv;
                break;

            case 'V': /* show version number */
                fmt::print(stderr, "{:s} {:s}\n", MyName, LONG_VERSION_STRING);
                exit(0);

            case 944:
                fmt::print("{:s}\n", std::empty(config.torrent_ids) ? "all" : config.torrent_ids.c_str());
                break;

            case TR_OPT_ERR:
                fmt::print(stderr, "invalid option\n");
                show_usage();
                status |= EXIT_FAILURE;
                break;

            case TR_OPT_UNK:
                if (auto* tadd_map = tadd.get_if<tr_variant::Map>(); tadd_map != nullptr)
                {
                    auto* const params = tadd_map->find_if<tr_variant::Map>(TR_KEY_params);
                    TR_ASSERT(params != nullptr);
                    if (params != nullptr)
                    {
                        if (auto const metainfo = get_encoded_metainfo(optarg); !std::empty(metainfo))
                        {
                            params->try_emplace(TR_KEY_metainfo, metainfo);
                        }
                        else
                        {
                            params->try_emplace(TR_KEY_filename, optarg_sv);
                        }
                    }
                }
                else
                {
                    fmt::print(stderr, "Unknown option: {:s}\n", optarg_sv);
                    status |= EXIT_FAILURE;
                }

                break;

            default:
                break;
            }
        }
        else if (step_mode == MODE_TORRENT_GET)
        {
            if (auto* tset_map = tset.get_if<tr_variant::Map>(); tset_map != nullptr)
            {
                auto* const params = tset_map->find_if<tr_variant::Map>(TR_KEY_params);
                TR_ASSERT(params != nullptr);
                if (params != nullptr)
                {
                    add_id_arg(*params, config);
                    status |= flush(rpcurl, &tset, config);
                }
            }

            auto map = tr_variant::Map{ 3U };
            map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
            map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_get));

            auto params = tr_variant::Map{ 1U };
            auto fields = tr_variant::Vector{};

            switch (c)
            {
            case 'F':
                config.filter = optarg_sv;
                map.insert_or_assign(TR_KEY_id, ID_FILTER);

                for (auto const& key : DetailsKeys)
                {
                    fields.emplace_back(tr_variant::unmanaged_string(key));
                }

                add_id_arg(params, config, "all");
                break;
            case 'i':
                map.insert_or_assign(TR_KEY_id, ID_DETAILS);

                for (auto const& key : DetailsKeys)
                {
                    fields.emplace_back(tr_variant::unmanaged_string(key));
                }

                add_id_arg(params, config);
                break;

            case 'l':
                map.insert_or_assign(TR_KEY_id, ID_LIST);

                for (auto const& key : ListKeys)
                {
                    fields.emplace_back(tr_variant::unmanaged_string(key));
                }

                add_id_arg(params, config, "all");
                break;

            case 940:
                map.insert_or_assign(TR_KEY_id, ID_FILES);

                for (auto const& key : FilesKeys)
                {
                    fields.emplace_back(tr_variant::unmanaged_string(key));
                }

                add_id_arg(params, config);
                break;

            case 941:
                map.insert_or_assign(TR_KEY_id, ID_PEERS);
                fields.emplace_back(tr_variant::unmanaged_string(TR_KEY_peers));
                add_id_arg(params, config);
                break;

            case 942:
                map.insert_or_assign(TR_KEY_id, ID_PIECES);
                fields.emplace_back(tr_variant::unmanaged_string(TR_KEY_pieces));
                fields.emplace_back(tr_variant::unmanaged_string(TR_KEY_piece_count));
                add_id_arg(params, config);
                break;

            case 943:
                map.insert_or_assign(TR_KEY_id, ID_TRACKERS);
                fields.emplace_back(tr_variant::unmanaged_string(TR_KEY_tracker_stats));
                add_id_arg(params, config);
                break;

            default:
                TR_ASSERT_MSG(false, "unhandled value");
                break;
            }

            params.insert_or_assign(TR_KEY_fields, std::move(fields));
            map.insert_or_assign(TR_KEY_params, std::move(params));
            auto top = tr_variant{ std::move(map) };
            status |= flush(rpcurl, &top, config);
        }
        else if (step_mode == MODE_SESSION_SET)
        {
            auto& args = ensure_sset(sset);

            switch (c)
            {
            case 800:
                args.insert_or_assign(TR_KEY_script_torrent_done_filename, optarg_sv);
                args.insert_or_assign(TR_KEY_script_torrent_done_enabled, true);
                break;

            case 801:
                args.insert_or_assign(TR_KEY_script_torrent_done_enabled, false);
                break;

            case 802:
                args.insert_or_assign(TR_KEY_script_torrent_done_seeding_filename, optarg_sv);
                args.insert_or_assign(TR_KEY_script_torrent_done_seeding_enabled, true);
                break;

            case 803:
                args.insert_or_assign(TR_KEY_script_torrent_done_seeding_enabled, false);
                break;

            case 970:
                args.insert_or_assign(TR_KEY_alt_speed_enabled, true);
                break;

            case 971:
                args.insert_or_assign(TR_KEY_alt_speed_enabled, false);
                break;

            case 972:
                args.insert_or_assign(TR_KEY_alt_speed_down, numarg(optarg_sv));
                break;

            case 973:
                args.insert_or_assign(TR_KEY_alt_speed_up, numarg(optarg_sv));
                break;

            case 974:
                args.insert_or_assign(TR_KEY_alt_speed_time_enabled, true);
                break;

            case 975:
                args.insert_or_assign(TR_KEY_alt_speed_time_enabled, false);
                break;

            case 976:
                add_time(args, TR_KEY_alt_speed_time_begin, optarg_sv);
                break;

            case 977:
                add_time(args, TR_KEY_alt_speed_time_end, optarg_sv);
                break;

            case 978:
                add_days(args, TR_KEY_alt_speed_time_day, optarg_sv);
                break;

            case 'c':
                args.insert_or_assign(TR_KEY_incomplete_dir, optarg_sv);
                args.insert_or_assign(TR_KEY_incomplete_dir_enabled, true);
                break;

            case 'C':
                args.insert_or_assign(TR_KEY_incomplete_dir_enabled, false);
                break;

            case 'e':
                args.insert_or_assign(TR_KEY_cache_size_mib, tr_num_parse<int64_t>(optarg_sv).value());
                break;

            case 910:
                args.insert_or_assign(TR_KEY_encryption, tr_variant::unmanaged_string("required"sv));
                break;

            case 911:
                args.insert_or_assign(TR_KEY_encryption, tr_variant::unmanaged_string("preferred"sv));
                break;

            case 912:
                args.insert_or_assign(TR_KEY_encryption, tr_variant::unmanaged_string("tolerated"sv));
                break;

            case 'm':
                args.insert_or_assign(TR_KEY_port_forwarding_enabled, true);
                break;

            case 'M':
                args.insert_or_assign(TR_KEY_port_forwarding_enabled, false);
                break;

            case 'o':
                args.insert_or_assign(TR_KEY_dht_enabled, true);
                break;

            case 'O':
                args.insert_or_assign(TR_KEY_dht_enabled, false);
                break;

            case 830:
                set_preferred_transports(args, optarg_sv);
                break;

            case 831:
                args.insert_or_assign(TR_KEY_utp_enabled, true);
                break;

            case 832:
                args.insert_or_assign(TR_KEY_utp_enabled, false);
                break;

            case 'p':
                args.insert_or_assign(TR_KEY_peer_port, numarg(optarg_sv));
                break;

            case 'P':
                args.insert_or_assign(TR_KEY_peer_port_random_on_start, true);
                break;

            case 'x':
                args.insert_or_assign(TR_KEY_pex_enabled, true);
                break;

            case 'X':
                args.insert_or_assign(TR_KEY_pex_enabled, false);
                break;

            case 'y':
                args.insert_or_assign(TR_KEY_lpd_enabled, true);
                break;

            case 'Y':
                args.insert_or_assign(TR_KEY_lpd_enabled, false);
                break;

            case 953:
                args.insert_or_assign(TR_KEY_seed_ratio_limit, tr_num_parse<double>(optarg_sv).value());
                args.insert_or_assign(TR_KEY_seed_ratio_limited, true);
                break;

            case 954:
                args.insert_or_assign(TR_KEY_seed_ratio_limited, false);
                break;

            case 958:
                args.insert_or_assign(TR_KEY_idle_seeding_limit, tr_num_parse<int64_t>(optarg_sv).value());
                args.insert_or_assign(TR_KEY_idle_seeding_limit_enabled, true);
                break;

            case 959:
                args.insert_or_assign(TR_KEY_idle_seeding_limit_enabled, false);
                break;

            case 990:
                args.insert_or_assign(TR_KEY_start_added_torrents, false);
                break;

            case 991:
                args.insert_or_assign(TR_KEY_start_added_torrents, true);
                break;

            case 992:
                args.insert_or_assign(TR_KEY_trash_original_torrent_files, true);
                break;

            case 993:
                args.insert_or_assign(TR_KEY_trash_original_torrent_files, false);
                break;

            default:
                TR_ASSERT_MSG(false, "unhandled value");
                break;
            }
        }
        else if (step_mode == (MODE_SESSION_SET | MODE_TORRENT_SET))
        {
            tr_variant::Map* targs = nullptr;
            tr_variant::Map* sargs = nullptr;

            if (!std::empty(config.torrent_ids))
            {
                targs = &ensure_tset(tset);
            }
            else
            {
                sargs = &ensure_sset(sset);
            }

            switch (c)
            {
            case 'd':
                if (targs != nullptr)
                {
                    targs->insert_or_assign(TR_KEY_download_limit, numarg(optarg_sv));
                    targs->insert_or_assign(TR_KEY_download_limited, true);
                }
                else
                {
                    sargs->insert_or_assign(TR_KEY_speed_limit_down, numarg(optarg_sv));
                    sargs->insert_or_assign(TR_KEY_speed_limit_down_enabled, true);
                }

                break;

            case 'D':
                if (targs != nullptr)
                {
                    targs->insert_or_assign(TR_KEY_download_limited, false);
                }
                else
                {
                    sargs->insert_or_assign(TR_KEY_speed_limit_down_enabled, false);
                }

                break;

            case 'u':
                if (targs != nullptr)
                {
                    targs->insert_or_assign(TR_KEY_upload_limit, numarg(optarg_sv));
                    targs->insert_or_assign(TR_KEY_upload_limited, true);
                }
                else
                {
                    sargs->insert_or_assign(TR_KEY_speed_limit_up, numarg(optarg_sv));
                    sargs->insert_or_assign(TR_KEY_speed_limit_up_enabled, true);
                }

                break;

            case 'U':
                if (targs != nullptr)
                {
                    targs->insert_or_assign(TR_KEY_upload_limited, false);
                }
                else
                {
                    sargs->insert_or_assign(TR_KEY_speed_limit_up_enabled, false);
                }

                break;

            case 930:
                if (targs != nullptr)
                {
                    targs->insert_or_assign(TR_KEY_peer_limit, tr_num_parse<int64_t>(optarg_sv).value());
                }
                else
                {
                    sargs->insert_or_assign(TR_KEY_peer_limit_global, tr_num_parse<int64_t>(optarg_sv).value());
                }

                break;

            default:
                TR_ASSERT_MSG(false, "unhandled value");
                break;
            }
        }
        else if (step_mode == MODE_TORRENT_SET)
        {
            tr_variant::Map& args = ensure_tset(tset);

            switch (c)
            {
            case 712:
                {
                    auto* list = args.find_if<tr_variant::Vector>(TR_KEY_tracker_remove);
                    if (list == nullptr)
                    {
                        list = args.insert_or_assign(TR_KEY_tracker_remove, tr_variant::make_vector(1))
                                   .first.get_if<tr_variant::Vector>();
                    }
                    list->emplace_back(tr_num_parse<int64_t>(optarg_sv).value());
                }
                break;

            case 950:
                args.insert_or_assign(TR_KEY_seed_ratio_limit, tr_num_parse<double>(optarg_sv).value());
                args.insert_or_assign(TR_KEY_seed_ratio_mode, TR_RATIOLIMIT_SINGLE);
                break;

            case 951:
                args.insert_or_assign(TR_KEY_seed_ratio_mode, TR_RATIOLIMIT_GLOBAL);
                break;

            case 952:
                args.insert_or_assign(TR_KEY_seed_ratio_mode, TR_RATIOLIMIT_UNLIMITED);
                break;

            case 955:
                args.insert_or_assign(TR_KEY_seed_idle_limit, tr_num_parse<int64_t>(optarg_sv).value());
                args.insert_or_assign(TR_KEY_seed_idle_mode, TR_IDLELIMIT_SINGLE);
                break;

            case 956:
                args.insert_or_assign(TR_KEY_seed_idle_mode, TR_IDLELIMIT_GLOBAL);
                break;

            case 957:
                args.insert_or_assign(TR_KEY_seed_idle_mode, TR_IDLELIMIT_UNLIMITED);
                break;

            case 984:
                args.insert_or_assign(TR_KEY_honors_session_limits, true);
                break;

            case 985:
                args.insert_or_assign(TR_KEY_honors_session_limits, false);
                break;

            default:
                TR_ASSERT_MSG(false, "unhandled value");
                break;
            }
        }
        else if (step_mode == (MODE_TORRENT_SET | MODE_TORRENT_ADD))
        {
            tr_variant::Map& args = tadd.has_value() ? ensure_tadd(tadd) : ensure_tset(tset);

            switch (c)
            {
            case 'g':
                args.insert_or_assign(TR_KEY_files_wanted, make_files_list(optarg_sv));
                break;

            case 'G':
                args.insert_or_assign(TR_KEY_files_unwanted, make_files_list(optarg_sv));
                break;

            case 'L':
                add_labels(args, optarg_sv);
                break;

            case 730:
                set_group(args, optarg_sv);
                break;

            case 731:
                set_group(args, ""sv);
                break;

            case 900:
                args.insert_or_assign(TR_KEY_priority_high, make_files_list(optarg_sv));
                break;

            case 901:
                args.insert_or_assign(TR_KEY_priority_normal, make_files_list(optarg_sv));
                break;

            case 902:
                args.insert_or_assign(TR_KEY_priority_low, make_files_list(optarg_sv));
                break;

            case 700:
                args.insert_or_assign(TR_KEY_bandwidth_priority, 1);
                break;

            case 701:
                args.insert_or_assign(TR_KEY_bandwidth_priority, 0);
                break;

            case 702:
                args.insert_or_assign(TR_KEY_bandwidth_priority, -1);
                break;

            case 710:
                {
                    auto* list = args.find_if<tr_variant::Vector>(TR_KEY_tracker_add);
                    if (list == nullptr)
                    {
                        list = args.insert_or_assign(TR_KEY_tracker_add, tr_variant::make_vector(1))
                                   .first.get_if<tr_variant::Vector>();
                    }
                    list->emplace_back(optarg_sv);
                }
                break;

            default:
                TR_ASSERT_MSG(false, "unhandled value");
                break;
            }
        }
        else if (step_mode == (MODE_SESSION_SET | MODE_TORRENT_SET | MODE_TORRENT_ADD))
        {
            tr_variant::Map& args = [&]() -> tr_variant::Map&
            {
                if (tadd.has_value())
                {
                    return ensure_tadd(tadd);
                }
                if (!std::empty(config.torrent_ids))
                {
                    return ensure_tset(tset);
                }
                return ensure_sset(sset);
            }();

            switch (c)
            {
            case 994:
                args.insert_or_assign(TR_KEY_sequential_download, true);
                if (optarg != nullptr)
                {
                    args.insert_or_assign(TR_KEY_sequential_download_from_piece, numarg(optarg_sv));
                }
                break;

            case 995:
                args.insert_or_assign(TR_KEY_sequential_download, false);
                break;

            default:
                TR_ASSERT_MSG(false, "unhandled value");
                break;
            }
        }
        else if (step_mode == MODE_TORRENT_REMOVE)
        {
            auto params = tr_variant::Map{ 2U };
            params.try_emplace(TR_KEY_delete_local_data, c == 840);
            add_id_arg(params, config);

            auto map = tr_variant::Map{ 3U };
            map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
            map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_remove));
            map.try_emplace(TR_KEY_params, std::move(params));

            auto top = tr_variant{ std::move(map) };
            status |= flush(rpcurl, &top, config);
        }
        else if (step_mode == MODE_TORRENT_START_STOP)
        {
            auto const is_stop = c == 'S';
            if (auto* tadd_map = tadd.get_if<tr_variant::Map>(); tadd_map != nullptr)
            {
                auto* const params = tadd_map->find_if<tr_variant::Map>(TR_KEY_params);
                TR_ASSERT(params != nullptr);
                if (params != nullptr)
                {
                    params->insert_or_assign(TR_KEY_paused, is_stop);
                }
            }
            else
            {
                auto params = tr_variant::Map{ 1U };
                add_id_arg(params, config);

                auto const key = is_stop ? TR_KEY_torrent_stop : TR_KEY_torrent_start;
                auto map = tr_variant::Map{ 3U };
                map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(key));
                map.try_emplace(TR_KEY_params, std::move(params));

                auto top = tr_variant{ std::move(map) };
                status |= flush(rpcurl, &top, config);
            }
        }
        else if (step_mode == MODE_TORRENT_ACTION)
        {
            static auto constexpr Method = [](int option)
            {
                switch (option)
                {
                case 'v':
                    return TR_KEY_torrent_verify;
                case 600:
                    return TR_KEY_torrent_reannounce;
                default:
                    TR_ASSERT_MSG(false, "unhandled value");
                    return TR_KEY_NONE;
                }
            };

            if (auto* tset_map = tset.get_if<tr_variant::Map>(); tset_map != nullptr)
            {
                auto* const params = tset_map->find_if<tr_variant::Map>(TR_KEY_params);
                TR_ASSERT(params != nullptr);
                if (params != nullptr)
                {
                    add_id_arg(*params, config);
                    status |= flush(rpcurl, &tset, config);
                }
            }

            auto params = tr_variant::Map{ 1U };
            add_id_arg(params, config);

            auto const key = Method(c);
            auto map = tr_variant::Map{ 3U };
            map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
            map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(key));
            map.try_emplace(TR_KEY_params, std::move(params));

            auto top = tr_variant{ std::move(map) };
            status |= flush(rpcurl, &top, config);
        }
        else
        {
            switch (c)
            {
            case 920: // session-info
                {
                    auto map = tr_variant::Map{ 3U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_session_get));
                    map.try_emplace(TR_KEY_id, ID_SESSION);

                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 'w':
                {
                    auto& args = tadd.has_value() ? ensure_tadd(tadd) : ensure_sset(sset);
                    args.insert_or_assign(TR_KEY_download_dir, optarg_sv);
                }
                break;

            case 850:
                {
                    auto map = tr_variant::Map{ 2U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_session_close));
                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 963:
                {
                    auto map = tr_variant::Map{ 3U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_blocklist_update));
                    map.try_emplace(TR_KEY_id, ID_BLOCKLIST);
                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 921:
                {
                    auto map = tr_variant::Map{ 3U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_session_stats));
                    map.try_emplace(TR_KEY_id, ID_STATS);
                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 962:
                {
                    auto map = tr_variant::Map{ 3U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_port_test));
                    map.try_emplace(TR_KEY_id, ID_PORTTEST);
                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 960:
                {
                    auto params = tr_variant::Map{ 3U };
                    params.try_emplace(TR_KEY_location, optarg_sv);
                    params.try_emplace(TR_KEY_move, true);
                    add_id_arg(params, config);

                    auto map = tr_variant::Map{ 4U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_set_location));
                    map.try_emplace(TR_KEY_params, std::move(params));
                    map.try_emplace(TR_KEY_id, ID_NOOP);

                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 961: /* set location */
                // TODO (5.0.0):
                // 1. Remove tadd.has_value() branch
                // 2. Group with --move under MODE_TORRENT_SET_LOCATION
                if (auto* tadd_map = tadd.get_if<tr_variant::Map>(); tadd_map != nullptr)
                {
                    auto* const params = tadd_map->find_if<tr_variant::Map>(TR_KEY_params);
                    TR_ASSERT(params != nullptr);
                    if (params != nullptr)
                    {
                        params->try_emplace(TR_KEY_download_dir, optarg_sv);
                    }
                }
                else
                {
                    auto params = tr_variant::Map{ 3U };
                    params.try_emplace(TR_KEY_location, optarg_sv);
                    params.try_emplace(TR_KEY_move, false);
                    add_id_arg(params, config);

                    auto map = tr_variant::Map{ 4U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_set_location));
                    map.try_emplace(TR_KEY_params, std::move(params));
                    map.try_emplace(TR_KEY_id, ID_NOOP);

                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            case 964:
                {
                    auto args = tr_variant::Map{ 3U };
                    args.try_emplace(TR_KEY_path, rename_from);
                    args.try_emplace(TR_KEY_name, optarg_sv);
                    add_id_arg(args, config);

                    auto map = tr_variant::Map{ 4U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_torrent_rename_path));
                    map.try_emplace(TR_KEY_params, std::move(args));
                    map.try_emplace(TR_KEY_id, ID_NOOP);

                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                    rename_from.clear();
                }
                break;

            case 965:
                rename_from = optarg_sv;
                break;

            case 732:
                {
                    auto map = tr_variant::Map{ 3U };
                    map.try_emplace(TR_KEY_jsonrpc, tr_variant::unmanaged_string(JsonRpc::Version));
                    map.try_emplace(TR_KEY_method, tr_variant::unmanaged_string(TR_KEY_group_get));
                    map.try_emplace(TR_KEY_id, ID_GROUPS);

                    auto top = tr_variant{ std::move(map) };
                    status |= flush(rpcurl, &top, config);
                }
                break;

            default:
                fmt::print(stderr, "got opt [{:d}]\n", c);
                show_usage();
                break;
            }
        }
    }

    if (tadd.has_value())
    {
        status |= flush(rpcurl, &tadd, config);
    }

    if (auto* tset_map = tset.get_if<tr_variant::Map>(); tset_map != nullptr)
    {
        auto* const params = tset_map->find_if<tr_variant::Map>(TR_KEY_params);
        TR_ASSERT(params != nullptr);
        if (params != nullptr)
        {
            add_id_arg(*params, config);
            status |= flush(rpcurl, &tset, config);
        }
    }

    if (sset.has_value())
    {
        status |= flush(rpcurl, &sset, config);
    }

    return status;
}

bool parse_port_string(std::string_view sv, uint16_t& port)
{
    auto remainder = std::string_view{};
    auto parsed = tr_num_parse<uint16_t>(sv, &remainder);
    auto ok = parsed && std::empty(remainder);
    if (ok)
    {
        port = *parsed;
    }

    return ok;
}

/* [host:port] or [host] or [port] or [http(s?)://host:port/transmission/] */
void get_host_and_port_and_rpc_url(
    int& argc,
    char** argv,
    std::string& host,
    uint16_t& port,
    std::string& rpcurl,
    RemoteConfig& config)
{
    if (*argv[1] == '-')
    {
        return;
    }

    auto const sv = std::string_view{ argv[1] };
    if (tr_strv_starts_with(sv, "http://")) /* user passed in http rpc url */
    {
        rpcurl = fmt::format("{:s}/rpc/", sv.substr(7));
    }
    else if (tr_strv_starts_with(sv, "https://")) /* user passed in https rpc url */
    {
        config.use_ssl = true;
        rpcurl = fmt::format("{:s}/rpc/", sv.substr(8));
    }
    else if (parse_port_string(sv, port))
    {
        // it was just a port
    }
    else if (auto const first_colon = sv.find(':'); first_colon == std::string_view::npos)
    {
        // it was a non-ipv6 host with no port
        host = sv;
    }
    else if (auto const last_colon = sv.rfind(':'); first_colon == last_colon)
    {
        // if only one colon, it's probably "$host:$port"
        if (parse_port_string(sv.substr(last_colon + 1), port))
        {
            host = sv.substr(0, last_colon);
        }
    }
    else
    {
        auto const is_unbracketed_ipv6 = !tr_strv_starts_with(sv, '[') && last_colon != std::string_view::npos;
        host = is_unbracketed_ipv6 ? fmt::format("[{:s}]", sv) : sv;
    }

    argc -= 1;

    for (int i = 1; i < argc; ++i)
    {
        argv[i] = argv[i + 1];
    }
}
} // namespace

int tr_main(int argc, char* argv[])
{
    tr_lib_init();

    tr_locale_set_global("");

    auto config = RemoteConfig{};
    auto port = DefaultPort;
    auto host = std::string{};
    auto rpcurl = std::string{};

    if (argc < 2)
    {
        show_usage();
        return EXIT_FAILURE;
    }

    get_host_and_port_and_rpc_url(argc, argv, host, port, rpcurl, config);

    if (std::empty(host))
    {
        host = DefaultHost;
    }

    if (std::empty(rpcurl))
    {
        rpcurl = fmt::format("{:s}:{:d}{:s}", host, port, DefaultUrl);
    }

    return process_args(rpcurl.c_str(), argc, (char const* const*)argv, config);
}
