/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster;

import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.ElasticsearchException;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.TimeoutClusterStateListener;
import org.elasticsearch.cluster.service.ClusterApplierService;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.util.concurrent.ThreadContext;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.threadpool.ThreadPool;

public class ClusterStateObserver {
    protected final Logger logger;
    private final Predicate<ClusterState> MATCH_ALL_CHANGES_PREDICATE = state -> true;
    private final ClusterApplierService clusterApplierService;
    private final ThreadPool threadPool;
    private final ThreadContext contextHolder;
    volatile TimeValue timeOutValue;
    final AtomicReference<StoredState> lastObservedState;
    final TimeoutClusterStateListener clusterStateListener = new ObserverClusterStateListener();
    final AtomicReference<ObservingContext> observingContext = new AtomicReference<Object>(null);
    volatile Long startTimeMS;
    volatile boolean timedOut;

    public ClusterStateObserver(ClusterService clusterService, Logger logger, ThreadContext contextHolder) {
        this(clusterService, new TimeValue(60000L), logger, contextHolder);
    }

    public ClusterStateObserver(ClusterService clusterService, @Nullable TimeValue timeout, Logger logger, ThreadContext contextHolder) {
        this(clusterService.state(), clusterService, timeout, logger, contextHolder);
    }

    public ClusterStateObserver(ClusterState initialState, ClusterService clusterService, @Nullable TimeValue timeout, Logger logger, ThreadContext contextHolder) {
        this(initialState, clusterService.getClusterApplierService(), timeout, logger, contextHolder);
    }

    public ClusterStateObserver(ClusterState initialState, ClusterApplierService clusterApplierService, @Nullable TimeValue timeout, Logger logger, ThreadContext contextHolder) {
        this.clusterApplierService = clusterApplierService;
        this.threadPool = clusterApplierService.threadPool();
        this.lastObservedState = new AtomicReference<StoredState>(new StoredState(initialState));
        this.timeOutValue = timeout;
        if (this.timeOutValue != null) {
            this.startTimeMS = this.threadPool.relativeTimeInMillis();
        }
        this.logger = logger;
        this.contextHolder = contextHolder;
    }

    public ClusterState setAndGetObservedState() {
        if (this.observingContext.get() != null) {
            throw new ElasticsearchException("cannot set current cluster state while waiting for a cluster state change", new Object[0]);
        }
        ClusterState clusterState = this.clusterApplierService.state();
        this.lastObservedState.set(new StoredState(clusterState));
        return clusterState;
    }

    public boolean isTimedOut() {
        return this.timedOut;
    }

    public void waitForNextChange(Listener listener) {
        this.waitForNextChange(listener, this.MATCH_ALL_CHANGES_PREDICATE);
    }

    public void waitForNextChange(Listener listener, @Nullable TimeValue timeOutValue) {
        this.waitForNextChange(listener, this.MATCH_ALL_CHANGES_PREDICATE, timeOutValue);
    }

    public void waitForNextChange(Listener listener, Predicate<ClusterState> statePredicate) {
        this.waitForNextChange(listener, statePredicate, null);
    }

    public void waitForNextChange(Listener listener, Predicate<ClusterState> statePredicate, @Nullable TimeValue timeOutValue) {
        Long timeoutTimeLeftMS;
        listener = new ContextPreservingListener(listener, this.contextHolder.newRestorableContext(false));
        if (this.observingContext.get() != null) {
            throw new ElasticsearchException("already waiting for a cluster state change", new Object[0]);
        }
        if (timeOutValue == null) {
            timeOutValue = this.timeOutValue;
            if (timeOutValue != null) {
                long timeSinceStartMS = this.threadPool.relativeTimeInMillis() - this.startTimeMS;
                timeoutTimeLeftMS = timeOutValue.millis() - timeSinceStartMS;
                if (timeoutTimeLeftMS <= 0L) {
                    this.logger.trace("observer timed out. notifying listener. timeout setting [{}], time since start [{}]", (Object)timeOutValue, (Object)new TimeValue(timeSinceStartMS));
                    this.timedOut = true;
                    this.lastObservedState.set(new StoredState(this.clusterApplierService.state()));
                    listener.onTimeout(timeOutValue);
                    return;
                }
            } else {
                timeoutTimeLeftMS = null;
            }
        } else {
            this.startTimeMS = this.threadPool.relativeTimeInMillis();
            this.timeOutValue = timeOutValue;
            timeoutTimeLeftMS = timeOutValue.millis();
            this.timedOut = false;
        }
        ClusterState newState = this.clusterApplierService.state();
        if (this.lastObservedState.get().isOlderOrDifferentMaster(newState) && statePredicate.test(newState)) {
            this.logger.trace("observer: sampled state accepted by predicate ({})", (Object)newState);
            this.lastObservedState.set(new StoredState(newState));
            listener.onNewClusterState(newState);
        } else {
            this.logger.trace("observer: sampled state rejected by predicate ({}). adding listener to ClusterService", (Object)newState);
            ObservingContext context = new ObservingContext(listener, statePredicate);
            if (!this.observingContext.compareAndSet(null, context)) {
                throw new ElasticsearchException("already waiting for a cluster state change", new Object[0]);
            }
            this.clusterApplierService.addTimeoutListener(timeoutTimeLeftMS == null ? null : new TimeValue(timeoutTimeLeftMS.longValue()), this.clusterStateListener);
        }
    }

    class ObserverClusterStateListener
    implements TimeoutClusterStateListener {
        ObserverClusterStateListener() {
        }

        @Override
        public void clusterChanged(ClusterChangedEvent event) {
            ObservingContext context = ClusterStateObserver.this.observingContext.get();
            if (context == null) {
                return;
            }
            ClusterState state = event.state();
            if (context.statePredicate.test(state)) {
                if (ClusterStateObserver.this.observingContext.compareAndSet(context, null)) {
                    ClusterStateObserver.this.clusterApplierService.removeTimeoutListener(this);
                    ClusterStateObserver.this.logger.trace("observer: accepting cluster state change ({})", (Object)state);
                    ClusterStateObserver.this.lastObservedState.set(new StoredState(state));
                    context.listener.onNewClusterState(state);
                } else {
                    ClusterStateObserver.this.logger.trace("observer: predicate approved change but observing context has changed - ignoring (new cluster state version [{}])", (Object)state.version());
                }
            } else {
                ClusterStateObserver.this.logger.trace("observer: predicate rejected change (new cluster state version [{}])", (Object)state.version());
            }
        }

        @Override
        public void postAdded() {
            ObservingContext context = ClusterStateObserver.this.observingContext.get();
            if (context == null) {
                return;
            }
            ClusterState newState = ClusterStateObserver.this.clusterApplierService.state();
            if (ClusterStateObserver.this.lastObservedState.get().isOlderOrDifferentMaster(newState) && context.statePredicate.test(newState)) {
                if (ClusterStateObserver.this.observingContext.compareAndSet(context, null)) {
                    ClusterStateObserver.this.logger.trace("observer: post adding listener: accepting current cluster state ({})", (Object)newState);
                    ClusterStateObserver.this.clusterApplierService.removeTimeoutListener(this);
                    ClusterStateObserver.this.lastObservedState.set(new StoredState(newState));
                    context.listener.onNewClusterState(newState);
                } else {
                    ClusterStateObserver.this.logger.trace("observer: postAdded - predicate approved state but observing context has changed - ignoring ({})", (Object)newState);
                }
            } else {
                ClusterStateObserver.this.logger.trace("observer: postAdded - predicate rejected state ({})", (Object)newState);
            }
        }

        @Override
        public void onClose() {
            ObservingContext context = ClusterStateObserver.this.observingContext.getAndSet(null);
            if (context != null) {
                ClusterStateObserver.this.logger.trace("observer: cluster service closed. notifying listener.");
                ClusterStateObserver.this.clusterApplierService.removeTimeoutListener(this);
                context.listener.onClusterServiceClose();
            }
        }

        @Override
        public void onTimeout(TimeValue timeout) {
            ObservingContext context = ClusterStateObserver.this.observingContext.getAndSet(null);
            if (context != null) {
                ClusterStateObserver.this.clusterApplierService.removeTimeoutListener(this);
                long timeSinceStartMS = ClusterStateObserver.this.threadPool.relativeTimeInMillis() - ClusterStateObserver.this.startTimeMS;
                ClusterStateObserver.this.logger.trace("observer: timeout notification from cluster service. timeout setting [{}], time since start [{}]", (Object)ClusterStateObserver.this.timeOutValue, (Object)new TimeValue(timeSinceStartMS));
                ClusterStateObserver.this.lastObservedState.set(new StoredState(ClusterStateObserver.this.clusterApplierService.state()));
                ClusterStateObserver.this.timedOut = true;
                context.listener.onTimeout(ClusterStateObserver.this.timeOutValue);
            }
        }

        public String toString() {
            return "ClusterStateObserver[" + ClusterStateObserver.this.observingContext.get() + "]";
        }
    }

    private static class StoredState {
        private final String masterNodeId;
        private final long version;

        StoredState(ClusterState clusterState) {
            this.masterNodeId = clusterState.nodes().getMasterNodeId();
            this.version = clusterState.version();
        }

        public boolean isOlderOrDifferentMaster(ClusterState clusterState) {
            return this.version < clusterState.version() || !Objects.equals(this.masterNodeId, clusterState.nodes().getMasterNodeId());
        }
    }

    public static interface Listener {
        public void onNewClusterState(ClusterState var1);

        public void onClusterServiceClose();

        public void onTimeout(TimeValue var1);
    }

    private static final class ContextPreservingListener
    implements Listener {
        private final Listener delegate;
        private final Supplier<ThreadContext.StoredContext> contextSupplier;

        private ContextPreservingListener(Listener delegate, Supplier<ThreadContext.StoredContext> contextSupplier) {
            this.contextSupplier = contextSupplier;
            this.delegate = delegate;
        }

        @Override
        public void onNewClusterState(ClusterState state) {
            try (ThreadContext.StoredContext context = this.contextSupplier.get();){
                this.delegate.onNewClusterState(state);
            }
        }

        @Override
        public void onClusterServiceClose() {
            try (ThreadContext.StoredContext context = this.contextSupplier.get();){
                this.delegate.onClusterServiceClose();
            }
        }

        @Override
        public void onTimeout(TimeValue timeout) {
            try (ThreadContext.StoredContext context = this.contextSupplier.get();){
                this.delegate.onTimeout(timeout);
            }
        }

        public String toString() {
            return "ContextPreservingListener[" + this.delegate + "]";
        }
    }

    static class ObservingContext {
        public final Listener listener;
        public final Predicate<ClusterState> statePredicate;

        ObservingContext(Listener listener, Predicate<ClusterState> statePredicate) {
            this.listener = listener;
            this.statePredicate = statePredicate;
        }

        public String toString() {
            return "ObservingContext[" + this.listener + "]";
        }
    }
}

