﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/b2bi/B2BIRequest.h>
#include <aws/b2bi/B2BI_EXPORTS.h>
#include <aws/b2bi/model/CapabilityOptions.h>
#include <aws/b2bi/model/Tag.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>

#include <utility>

namespace Aws {
namespace B2BI {
namespace Model {

/**
 */
class CreatePartnershipRequest : public B2BIRequest {
 public:
  AWS_B2BI_API CreatePartnershipRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreatePartnership"; }

  AWS_B2BI_API Aws::String SerializePayload() const override;

  AWS_B2BI_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>Specifies the unique, system-generated identifier for the profile connected
   * to this partnership.</p>
   */
  inline const Aws::String& GetProfileId() const { return m_profileId; }
  inline bool ProfileIdHasBeenSet() const { return m_profileIdHasBeenSet; }
  template <typename ProfileIdT = Aws::String>
  void SetProfileId(ProfileIdT&& value) {
    m_profileIdHasBeenSet = true;
    m_profileId = std::forward<ProfileIdT>(value);
  }
  template <typename ProfileIdT = Aws::String>
  CreatePartnershipRequest& WithProfileId(ProfileIdT&& value) {
    SetProfileId(std::forward<ProfileIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a descriptive name for the partnership.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreatePartnershipRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the email address associated with this trading partner.</p>
   */
  inline const Aws::String& GetEmail() const { return m_email; }
  inline bool EmailHasBeenSet() const { return m_emailHasBeenSet; }
  template <typename EmailT = Aws::String>
  void SetEmail(EmailT&& value) {
    m_emailHasBeenSet = true;
    m_email = std::forward<EmailT>(value);
  }
  template <typename EmailT = Aws::String>
  CreatePartnershipRequest& WithEmail(EmailT&& value) {
    SetEmail(std::forward<EmailT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the phone number associated with the partnership.</p>
   */
  inline const Aws::String& GetPhone() const { return m_phone; }
  inline bool PhoneHasBeenSet() const { return m_phoneHasBeenSet; }
  template <typename PhoneT = Aws::String>
  void SetPhone(PhoneT&& value) {
    m_phoneHasBeenSet = true;
    m_phone = std::forward<PhoneT>(value);
  }
  template <typename PhoneT = Aws::String>
  CreatePartnershipRequest& WithPhone(PhoneT&& value) {
    SetPhone(std::forward<PhoneT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies a list of the capabilities associated with this partnership.</p>
   */
  inline const Aws::Vector<Aws::String>& GetCapabilities() const { return m_capabilities; }
  inline bool CapabilitiesHasBeenSet() const { return m_capabilitiesHasBeenSet; }
  template <typename CapabilitiesT = Aws::Vector<Aws::String>>
  void SetCapabilities(CapabilitiesT&& value) {
    m_capabilitiesHasBeenSet = true;
    m_capabilities = std::forward<CapabilitiesT>(value);
  }
  template <typename CapabilitiesT = Aws::Vector<Aws::String>>
  CreatePartnershipRequest& WithCapabilities(CapabilitiesT&& value) {
    SetCapabilities(std::forward<CapabilitiesT>(value));
    return *this;
  }
  template <typename CapabilitiesT = Aws::String>
  CreatePartnershipRequest& AddCapabilities(CapabilitiesT&& value) {
    m_capabilitiesHasBeenSet = true;
    m_capabilities.emplace_back(std::forward<CapabilitiesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specify the structure that contains the details for the associated
   * capabilities.</p>
   */
  inline const CapabilityOptions& GetCapabilityOptions() const { return m_capabilityOptions; }
  inline bool CapabilityOptionsHasBeenSet() const { return m_capabilityOptionsHasBeenSet; }
  template <typename CapabilityOptionsT = CapabilityOptions>
  void SetCapabilityOptions(CapabilityOptionsT&& value) {
    m_capabilityOptionsHasBeenSet = true;
    m_capabilityOptions = std::forward<CapabilityOptionsT>(value);
  }
  template <typename CapabilityOptionsT = CapabilityOptions>
  CreatePartnershipRequest& WithCapabilityOptions(CapabilityOptionsT&& value) {
    SetCapabilityOptions(std::forward<CapabilityOptionsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Reserved for future use.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreatePartnershipRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the key-value pairs assigned to ARNs that you can use to group and
   * search for resources by type. You can attach this metadata to resources
   * (capabilities, partnerships, and so on) for any purpose.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreatePartnershipRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreatePartnershipRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_profileId;
  bool m_profileIdHasBeenSet = false;

  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  Aws::String m_email;
  bool m_emailHasBeenSet = false;

  Aws::String m_phone;
  bool m_phoneHasBeenSet = false;

  Aws::Vector<Aws::String> m_capabilities;
  bool m_capabilitiesHasBeenSet = false;

  CapabilityOptions m_capabilityOptions;
  bool m_capabilityOptionsHasBeenSet = false;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_clientTokenHasBeenSet = true;

  Aws::Vector<Tag> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace B2BI
}  // namespace Aws
