﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cleanrooms/CleanRoomsRequest.h>
#include <aws/cleanrooms/CleanRooms_EXPORTS.h>
#include <aws/cleanrooms/model/ComputeConfiguration.h>
#include <aws/cleanrooms/model/ProtectedQueryResultConfiguration.h>
#include <aws/cleanrooms/model/ProtectedQuerySQLParameters.h>
#include <aws/cleanrooms/model/ProtectedQueryType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CleanRooms {
namespace Model {

/**
 */
class StartProtectedQueryRequest : public CleanRoomsRequest {
 public:
  AWS_CLEANROOMS_API StartProtectedQueryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "StartProtectedQuery"; }

  AWS_CLEANROOMS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The type of the protected query to be started.</p>
   */
  inline ProtectedQueryType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(ProtectedQueryType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline StartProtectedQueryRequest& WithType(ProtectedQueryType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique identifier for the membership to run this query against. Currently
   * accepts a membership ID.</p>
   */
  inline const Aws::String& GetMembershipIdentifier() const { return m_membershipIdentifier; }
  inline bool MembershipIdentifierHasBeenSet() const { return m_membershipIdentifierHasBeenSet; }
  template <typename MembershipIdentifierT = Aws::String>
  void SetMembershipIdentifier(MembershipIdentifierT&& value) {
    m_membershipIdentifierHasBeenSet = true;
    m_membershipIdentifier = std::forward<MembershipIdentifierT>(value);
  }
  template <typename MembershipIdentifierT = Aws::String>
  StartProtectedQueryRequest& WithMembershipIdentifier(MembershipIdentifierT&& value) {
    SetMembershipIdentifier(std::forward<MembershipIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The protected SQL query parameters.</p>
   */
  inline const ProtectedQuerySQLParameters& GetSqlParameters() const { return m_sqlParameters; }
  inline bool SqlParametersHasBeenSet() const { return m_sqlParametersHasBeenSet; }
  template <typename SqlParametersT = ProtectedQuerySQLParameters>
  void SetSqlParameters(SqlParametersT&& value) {
    m_sqlParametersHasBeenSet = true;
    m_sqlParameters = std::forward<SqlParametersT>(value);
  }
  template <typename SqlParametersT = ProtectedQuerySQLParameters>
  StartProtectedQueryRequest& WithSqlParameters(SqlParametersT&& value) {
    SetSqlParameters(std::forward<SqlParametersT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details needed to write the query results.</p>
   */
  inline const ProtectedQueryResultConfiguration& GetResultConfiguration() const { return m_resultConfiguration; }
  inline bool ResultConfigurationHasBeenSet() const { return m_resultConfigurationHasBeenSet; }
  template <typename ResultConfigurationT = ProtectedQueryResultConfiguration>
  void SetResultConfiguration(ResultConfigurationT&& value) {
    m_resultConfigurationHasBeenSet = true;
    m_resultConfiguration = std::forward<ResultConfigurationT>(value);
  }
  template <typename ResultConfigurationT = ProtectedQueryResultConfiguration>
  StartProtectedQueryRequest& WithResultConfiguration(ResultConfigurationT&& value) {
    SetResultConfiguration(std::forward<ResultConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The compute configuration for the protected query.</p>
   */
  inline const ComputeConfiguration& GetComputeConfiguration() const { return m_computeConfiguration; }
  inline bool ComputeConfigurationHasBeenSet() const { return m_computeConfigurationHasBeenSet; }
  template <typename ComputeConfigurationT = ComputeConfiguration>
  void SetComputeConfiguration(ComputeConfigurationT&& value) {
    m_computeConfigurationHasBeenSet = true;
    m_computeConfiguration = std::forward<ComputeConfigurationT>(value);
  }
  template <typename ComputeConfigurationT = ComputeConfiguration>
  StartProtectedQueryRequest& WithComputeConfiguration(ComputeConfigurationT&& value) {
    SetComputeConfiguration(std::forward<ComputeConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  ProtectedQueryType m_type{ProtectedQueryType::NOT_SET};
  bool m_typeHasBeenSet = false;

  Aws::String m_membershipIdentifier;
  bool m_membershipIdentifierHasBeenSet = false;

  ProtectedQuerySQLParameters m_sqlParameters;
  bool m_sqlParametersHasBeenSet = false;

  ProtectedQueryResultConfiguration m_resultConfiguration;
  bool m_resultConfigurationHasBeenSet = false;

  ComputeConfiguration m_computeConfiguration;
  bool m_computeConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace CleanRooms
}  // namespace Aws
