﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cloudformation/CloudFormationRequest.h>
#include <aws/cloudformation/CloudFormation_EXPORTS.h>
#include <aws/cloudformation/model/RegistryType.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace CloudFormation {
namespace Model {

/**
 */
class DeregisterTypeRequest : public CloudFormationRequest {
 public:
  AWS_CLOUDFORMATION_API DeregisterTypeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DeregisterType"; }

  AWS_CLOUDFORMATION_API Aws::String SerializePayload() const override;

 protected:
  AWS_CLOUDFORMATION_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the extension.</p> <p>Conditional: You must
   * specify either <code>TypeName</code> and <code>Type</code>, or
   * <code>Arn</code>.</p>
   */
  inline const Aws::String& GetArn() const { return m_arn; }
  inline bool ArnHasBeenSet() const { return m_arnHasBeenSet; }
  template <typename ArnT = Aws::String>
  void SetArn(ArnT&& value) {
    m_arnHasBeenSet = true;
    m_arn = std::forward<ArnT>(value);
  }
  template <typename ArnT = Aws::String>
  DeregisterTypeRequest& WithArn(ArnT&& value) {
    SetArn(std::forward<ArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The kind of extension.</p> <p>Conditional: You must specify either
   * <code>TypeName</code> and <code>Type</code>, or <code>Arn</code>.</p>
   */
  inline RegistryType GetType() const { return m_type; }
  inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
  inline void SetType(RegistryType value) {
    m_typeHasBeenSet = true;
    m_type = value;
  }
  inline DeregisterTypeRequest& WithType(RegistryType value) {
    SetType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the extension.</p> <p>Conditional: You must specify either
   * <code>TypeName</code> and <code>Type</code>, or <code>Arn</code>.</p>
   */
  inline const Aws::String& GetTypeName() const { return m_typeName; }
  inline bool TypeNameHasBeenSet() const { return m_typeNameHasBeenSet; }
  template <typename TypeNameT = Aws::String>
  void SetTypeName(TypeNameT&& value) {
    m_typeNameHasBeenSet = true;
    m_typeName = std::forward<TypeNameT>(value);
  }
  template <typename TypeNameT = Aws::String>
  DeregisterTypeRequest& WithTypeName(TypeNameT&& value) {
    SetTypeName(std::forward<TypeNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of a specific version of the extension. The version ID is the value at
   * the end of the Amazon Resource Name (ARN) assigned to the extension version when
   * it is registered.</p>
   */
  inline const Aws::String& GetVersionId() const { return m_versionId; }
  inline bool VersionIdHasBeenSet() const { return m_versionIdHasBeenSet; }
  template <typename VersionIdT = Aws::String>
  void SetVersionId(VersionIdT&& value) {
    m_versionIdHasBeenSet = true;
    m_versionId = std::forward<VersionIdT>(value);
  }
  template <typename VersionIdT = Aws::String>
  DeregisterTypeRequest& WithVersionId(VersionIdT&& value) {
    SetVersionId(std::forward<VersionIdT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_arn;
  bool m_arnHasBeenSet = false;

  RegistryType m_type{RegistryType::NOT_SET};
  bool m_typeHasBeenSet = false;

  Aws::String m_typeName;
  bool m_typeNameHasBeenSet = false;

  Aws::String m_versionId;
  bool m_versionIdHasBeenSet = false;
};

}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
