﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/datasync/DataSync_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DataSync {
namespace Model {

/**
 * <p>The subnet and security groups that DataSync uses to connect to one of your
 * Amazon EFS file system's <a
 * href="https://docs.aws.amazon.com/efs/latest/ug/accessing-fs.html">mount
 * targets</a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/Ec2Config">AWS
 * API Reference</a></p>
 */
class Ec2Config {
 public:
  AWS_DATASYNC_API Ec2Config() = default;
  AWS_DATASYNC_API Ec2Config(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATASYNC_API Ec2Config& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATASYNC_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Specifies the ARN of a subnet where DataSync creates the <a
   * href="https://docs.aws.amazon.com/datasync/latest/userguide/datasync-network.html#required-network-interfaces">network
   * interfaces</a> for managing traffic during your transfer.</p> <p>The subnet must
   * be located:</p> <ul> <li> <p>In the same virtual private cloud (VPC) as the
   * Amazon EFS file system.</p> </li> <li> <p>In the same Availability Zone as at
   * least one mount target for the Amazon EFS file system.</p> </li> </ul>
   * <p>You don't need to specify a subnet that includes a file system mount
   * target.</p>
   */
  inline const Aws::String& GetSubnetArn() const { return m_subnetArn; }
  inline bool SubnetArnHasBeenSet() const { return m_subnetArnHasBeenSet; }
  template <typename SubnetArnT = Aws::String>
  void SetSubnetArn(SubnetArnT&& value) {
    m_subnetArnHasBeenSet = true;
    m_subnetArn = std::forward<SubnetArnT>(value);
  }
  template <typename SubnetArnT = Aws::String>
  Ec2Config& WithSubnetArn(SubnetArnT&& value) {
    SetSubnetArn(std::forward<SubnetArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the Amazon Resource Names (ARNs) of the security groups associated
   * with an Amazon EFS file system's mount target.</p>
   */
  inline const Aws::Vector<Aws::String>& GetSecurityGroupArns() const { return m_securityGroupArns; }
  inline bool SecurityGroupArnsHasBeenSet() const { return m_securityGroupArnsHasBeenSet; }
  template <typename SecurityGroupArnsT = Aws::Vector<Aws::String>>
  void SetSecurityGroupArns(SecurityGroupArnsT&& value) {
    m_securityGroupArnsHasBeenSet = true;
    m_securityGroupArns = std::forward<SecurityGroupArnsT>(value);
  }
  template <typename SecurityGroupArnsT = Aws::Vector<Aws::String>>
  Ec2Config& WithSecurityGroupArns(SecurityGroupArnsT&& value) {
    SetSecurityGroupArns(std::forward<SecurityGroupArnsT>(value));
    return *this;
  }
  template <typename SecurityGroupArnsT = Aws::String>
  Ec2Config& AddSecurityGroupArns(SecurityGroupArnsT&& value) {
    m_securityGroupArnsHasBeenSet = true;
    m_securityGroupArns.emplace_back(std::forward<SecurityGroupArnsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_subnetArn;
  bool m_subnetArnHasBeenSet = false;

  Aws::Vector<Aws::String> m_securityGroupArns;
  bool m_securityGroupArnsHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataSync
}  // namespace Aws
