/*-
 * Copyright (c) 2003 Lev Walkin <vlm@lionet.info>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: psrc-ulog.c,v 1.16 2004/07/27 09:49:00 vlm Exp $
 */

#include "ipcad.h"
#include "opt.h"
#include "cfgvar.h"
#include <assert.h>

#ifndef	PSRC_ulog
int
init_packet_source_ulog(packet_source_t *ps, int retry_mode) {
	(void)ps;
	(void)retry_mode;
	fprintf(stderr, "Linux ULOG source support "
		"is not properly compiled in "
		"or not supported by the target OS, "
		"please contact the ipcad vendor.\n");
	errno = EINVAL;
	return -1;
}

#else	/* !PSRC_ulog */

int
init_packet_source_ulog(packet_source_t *ps, int retry_mode) {
	int ulog_fd;
	struct sockaddr_nl nls;

	assert(ps->state != PST_INVALID);	/* Embryonic or Ready */
	assert(ps->iface_type == IFACE_ULOG);	/* Don't cook crap */

	/*
	 * Open Linux iptables ULOG file descriptor.
	 */
	ulog_fd = ps->fd;
	if(ulog_fd == -1) {
		ulog_fd = socket(PF_NETLINK, SOCK_RAW, NETLINK_NFLOG);
		if(ulog_fd == -1) return -1;
	} else {
		ps->state = PST_EMBRYONIC;
	}
	
	/*
	 * Bind to the ULOG packet source.
	 */
	memset(&nls, 0, sizeof(nls));
	nls.nl_family = AF_NETLINK;
	nls.nl_pid = getpid();
	nls.nl_groups = ps->iface.ulog.groupmask;

	if(bind(ulog_fd, (struct sockaddr *)&nls, sizeof(nls))) {
		if(ps->fd == -1)
			close(ulog_fd);
		return -1;
	}
	
	memset(&ps->iface.ulog.peer, 0, sizeof(ps->iface.ulog.peer)); 
	ps->iface.ulog.peer.nl_family = AF_NETLINK;
	ps->iface.ulog.peer.nl_pid = 0;
	ps->iface.ulog.peer.nl_groups = ps->iface.ulog.groupmask;

	ps->dlt = DLT_RAW;

    	if(ps->buf == NULL) {
		if(conf->bufsize) {
			ps->bufsize = conf->bufsize;
			if(ps->bufsize < 4096)
				/* Minimum value */
				ps->bufsize = 4096;
		} else {
			/* Reasonable default */
			ps->bufsize = 65536;
		}
		assert(ps->bufsize >= sizeof(struct nlmsgerr));
		assert(ps->bufsize >= sizeof(struct nlmsghdr));
		ps->buf = (char *)malloc(ps->bufsize);
		if(ps->buf == NULL) {
			if(ps->fd == -1)
				close(ulog_fd);
			/* ENOMEM */
			return -1;
		}
	}

	/*
	 * ULOG interface has really convoluted binding
	 * scheme, requiring the getpid(). IPCAD is a
	 * threaded application, so under Linux pid may change.
	 * Defer everything until we're running inside the worker thread.
	 */
	if(!retry_mode) {
		if(ps->fd == -1)
			close(ulog_fd);
		errno = ENETDOWN;
		return -1;
	}

	/*
	 * Finish initialization of the structure.
	 */

	ps->fd = ulog_fd;
	ps->state = PST_READY;
	
	return 0;
}

void
print_stats_ulog(FILE *f, packet_source_t *ps) {

	assert(ps->iface_type == IFACE_ULOG);

	fprintf(f, "Interface %s: received ??", IFNameBySource(ps));

	fprintf(f, ", %.0f m average %lld bytes/sec, %lld pkts/sec",
		ps->avg_period / 60,
		ps->bps_lp,
		ps->pps_lp
	);
	
	fprintf(f, ", dropped ??"); 
	fprintf(f, "\n");
}

#endif	/* PSRC_ulog */
