/* error location handling
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "stacksg.h"
#include "jalcomp.h"
#include "cstringf.h"

/* data which describes a source location */
char e_string[] = "(nowhere)";
loc_r nowhere_data = { e_string, e_string, 0, 0, 0 };
loc_t nowhere = &nowhere_data;

/* return allocated string which describes location loc */
char *location_string(loc_t loc)
{
    string s;
    stack_guard;
    if (loc == NULL) {
        sprintf(s, "");
    } else {
        sprintf(s, ">  file '%s' line %d char %d\n>  :  %s\n>  :  %s%s\n", loc->file_name,
                loc->line_nr, loc->token_start, loc->line, bar(loc->token_start - 1, '-'),
                bar(loc->token_size, '^')
            );
    }
    return new_string(s);
}

/* short version of the above */
char *loc_string(loc_t loc)
{
    string s;
    stack_guard;
    if (loc == NULL) {
        sprintf(s, "");
    } else {
        sprintf(s, "%04d:%02d %s ", loc->line_nr, loc->token_start, loc->file_name);
    }
    return new_string(s);
}


/********** logging and error handling **********/

/* log with build-in sprintf */
void flog(char *file, int line, char *m)
{
    stack_guard;
    printf("[%s at %d] %s \n", file, line, m);
    fflush(NULL);
}

string _log_string;
#define log(x) { \
   char *m=_log_string; sprintf x; flog( __FILE__, __LINE__, m ); }
#define trace  { \
   flog( __FILE__, __LINE__, "trace" ); }

/* report a fatal error (with context) and exit */
void ffatal(loc_t loc, char *m)
{
    stack_guard;
    printf(">  error: %s%40s\n", m, "");
    printf(location_string(loc));
/*   fflush( NULL ); */

    /* real error or an expected error? */
    if (catch_line != 0) {
        if ((loc == NULL)
            || (catch_line != loc->line_nr)
            || (catch_pos != loc->token_start)
            ) {
            printf(">  error expected at %d:%d\n", catch_line, catch_pos);
            free_all();
            exit(-1);
        } else {
            printf("error caught as expected\n");
            free_all();
            exit(0);
        }
    } else {
        if (verbose_stack)
            stack_dump();
        free_all();
        exit(-1);
    }
}

/* idem with build-in sprintf */
#define fatal( loc, x ) { string m; \
   sprintf x; \
   ffatal( loc, m ); \
}

/* fatal using the current context */
#define cfatal( x ) fatal( scanner_context->loc, x )

/* check an internal assumption */
void assert_fail_f(loc_t loc, char *file, int line)
{
    string m;
    sprintf(m, "assertion failure in %s line %d", file, line);
    ffatal(loc, m);
}

#define pre_check 12345678
#define post_check 56789012
typedef struct {
    long int check;
    int size;
    void *next;
} pre_block;
typedef struct {
    long int check;
} post_block;

pre_block *block_root = NULL;

/* check whether a pointer points to an allocated block */
void assert_pointer_p(loc_t loc, char *s, int n, void *p)
{
    string m, m2;
    stack_guard;
    if (p == NULL) {
        sprintf(m, "NULL pointer");
    } else if (!check_blocks) {
        return;
    } else {
        pre_block *pre = (pre_block *) (((char *) p) - sizeof(pre_block));
        post_block *post;
        post = (post_block *) (((char *) p) + pre->size);
        if (pre->check == pre_check) {
            if (post->check == post_check) {
                return;
            } else {
                sprintf(m, "SIZE or POST overwritten");
            }
        } else {
            sprintf(m, "PRE overwritten");
        }
    }
    sprintf(m2, "%s in %s line %d", m, s, n);
    ffatal(loc, m2);
}
void assert_arena_2(char *s, int n)
{
    pre_block *q;
    stack_guard;
    for (q = block_root; q != NULL; q = q->next) {
        assert_pointer_p(NULL, s, n, ((char *) q) + sizeof(pre_block));
    }
}
void assert_pointer_2(loc_t loc, char *s, int n, void *p)
{
    stack_guard;
    assert_pointer_p(loc, s, n, p);
    if (check_pool)
        assert_arena_2(s, n);
}

#define assert_chain { \
   assert_arena_2( __FILE__, __LINE__, p ); \
}
#define assert_pointer( loc, p ){ \
   if( check_asserts ){ \
      assert_pointer_2( loc, __FILE__, __LINE__, p ); \
   } \
}

#define MAX_ALLOC 10000
int allocated = 0;
void *alloc_list[MAX_ALLOC];

/* checked malloc */
int total_memory = 0;
#define allocation_chunk ( 64 * 1024 )
char *allocation_buffer;
int allocation_buffer_size = 0;
void *get_memory(int size)
{
    void *p;
    size = (size + 15) & 0xFFFFF0;
    if (size > allocation_chunk) {
        p = malloc(size);
        return p;
    } else if (allocation_buffer_size <= size) {
        allocation_buffer = malloc(allocation_chunk);
        allocation_buffer_size = allocation_chunk;
    }
    p = allocation_buffer;
    allocation_buffer += size;
    allocation_buffer_size -= size;
    return p;
}

void free_all(void)
{
    allocated--;
    while (allocated >= 0) {
        free(alloc_list[allocated]);
        allocated--;
    }
}

void *_allocate(int size, int line)
{
    int i;
    stack_guard;
    if (verbose_malloc)
        log((m, "%d allocate %d", line, (int) size));

    if (check_blocks) {
        post_block *post;
        int n = size + sizeof(pre_block) + sizeof(post_block);
        pre_block *pre = get_memory(n);

        if (pre == NULL) {
            fatal(NULL, (m, "out of memory"));
        }
        for (i = 0; i < n; i++) {
            *(i + (char *) pre) = 0;
        }

        pre->check = pre_check;
        pre->size = size;
        post = (post_block *)
            (((char *) pre) + sizeof(pre_block) + size);
        post->check = post_check;
        pre->next = block_root;
        block_root = pre;

        total_memory += n;
        return ((char *) pre) + sizeof(pre_block);
    } else {
        void *p = get_memory(size);

        if (p == NULL) {
            fatal(NULL, (m, "out of memory"));
        }
        if (check_memory_zero)
            for (i = 0; i < size; i++) {
                *(i + (char *) p) = 0;
            }
        total_memory += size;
        return p;
    }
}
