/*
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <string.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include "sg-util.h"
#include "st-category-bag.h"
#include "st-category-view.h"
#include "st-handler.h"
#include "st-statusbar.h"

/*** type definitions ********************************************************/

enum {
  PROP_0,
  PROP_TASK_THREAD
};

struct _STCategoryBagPrivate
{
  STThread		*task_thread;
};

/*** variable declarations ***************************************************/

static GObjectClass *parent_class = NULL;

/*** function declarations ***************************************************/

static void st_category_bag_class_init (STCategoryBagClass *class);
static void st_category_bag_init (STCategoryBag *bag);
static void st_category_bag_finalize (GObject *object);
static void st_category_bag_set_property (GObject *object,
					  unsigned int prop_id,
					  const GValue *value,
					  GParamSpec *pspec);
static void st_category_bag_get_property (GObject *object,
					  unsigned int prop_id,
					  GValue *value,
					  GParamSpec *pspec);

/*** implementation **********************************************************/

GType
st_category_bag_get_type (void)
{
  static GType category_bag_type = 0;
  
  if (! category_bag_type)
    {
      static const GTypeInfo category_bag_info = {
	sizeof(STCategoryBagClass),
	NULL,
	NULL,
	(GClassInitFunc) st_category_bag_class_init,
	NULL,
	NULL,
	sizeof(STCategoryBag),
	0,
	(GInstanceInitFunc) st_category_bag_init,
      };
      
      category_bag_type = g_type_register_static(G_TYPE_OBJECT,
						 "STCategoryBag",
						 &category_bag_info,
						 0);
    }

  return category_bag_type;
}

static void
st_category_bag_class_init (STCategoryBagClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  g_type_class_add_private(class, sizeof(STCategoryBagPrivate));

  object_class->finalize = st_category_bag_finalize;
  object_class->set_property = st_category_bag_set_property;
  object_class->get_property = st_category_bag_get_property;

  g_object_class_install_property(object_class,
				  PROP_TASK_THREAD,
				  g_param_spec_pointer("task-thread",
						       NULL,
						       NULL,
						       G_PARAM_READWRITE));
}

static void
st_category_bag_init (STCategoryBag *bag)
{
  bag->category = NULL;
  bag->handler = NULL;
  memset(&bag->iter, 0, sizeof(bag->iter));

  bag->statusbar = st_statusbar_new();
  g_object_ref(bag->statusbar);

  bag->statusbar_context_count = gtk_statusbar_get_context_id(GTK_STATUSBAR(bag->statusbar), _("Count messages"));
  bag->statusbar_context_task = gtk_statusbar_get_context_id(GTK_STATUSBAR(bag->statusbar), _("Task messages"));
  
  bag->priv = G_TYPE_INSTANCE_GET_PRIVATE(bag, ST_TYPE_CATEGORY_BAG, STCategoryBagPrivate);
}

static void
st_category_bag_finalize (GObject *object)
{
  STCategoryBag *bag = ST_CATEGORY_BAG(object);

  if (bag->category)
    st_handler_event_category_free(bag->handler, bag->category);

  g_object_unref(bag->statusbar);

  parent_class->finalize(object);
}

static void
st_category_bag_set_property (GObject *object,
			      unsigned int prop_id,
			      const GValue *value,
			      GParamSpec *pspec)
{
  STCategoryBag *bag = ST_CATEGORY_BAG(object);

  switch (prop_id)
    {
    case PROP_TASK_THREAD:
      bag->priv->task_thread = g_value_get_pointer(value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

static void
st_category_bag_get_property (GObject *object,
			      unsigned int prop_id,
			      GValue *value,
			      GParamSpec *pspec)
{
  STCategoryBag *bag = ST_CATEGORY_BAG(object);

  switch (prop_id)
    {
    case PROP_TASK_THREAD:
      g_value_set_pointer(value, bag->priv->task_thread);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

STCategoryBag *
st_category_bag_new (STHandler *handler)
{
  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);

  return st_category_bag_new_from_category(handler, st_handler_event_category_new(handler));
}

STCategoryBag *
st_category_bag_new_from_category (STHandler *handler, STCategory *category)
{
  STCategoryBag *bag;

  g_return_val_if_fail(ST_IS_HANDLER(handler), NULL);
  g_return_val_if_fail(category != NULL, NULL);

  bag = g_object_new(ST_TYPE_CATEGORY_BAG, NULL);

  bag->category = category;
  bag->handler = handler;

  return bag;
}

gboolean
st_category_bag_apply_url_cb (STCategoryBag *bag)
{
  g_return_val_if_fail(ST_IS_CATEGORY_BAG(bag), FALSE);

  if (ST_CATEGORY(bag)->url_cb)
    {
      if (! ST_CATEGORY(bag)->url_cb(ST_CATEGORY(bag)))
	return FALSE;
    }

  return TRUE;
}

void
st_category_bag_select (STCategoryBag *bag)
{
  STCategoryBag *selected_bag;

  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  selected_bag = st_handler_get_selected_category(bag->handler);
  if (selected_bag)
    {
      SG_UNSET_FLAGS(selected_bag->flags, ST_CATEGORY_BAG_SELECTED);
      g_object_unref(selected_bag);
    }

  SG_SET_FLAGS(bag->flags, ST_CATEGORY_BAG_SELECTED);
}

void
st_category_bag_set_task_thread (STCategoryBag *bag, STThread *task_thread)
{
  g_return_if_fail(ST_IS_CATEGORY_BAG(bag));

  g_object_set(bag, "task-thread", task_thread, NULL);
}

STThread *
st_category_bag_get_task_thread (STCategoryBag *bag)
{
  g_return_val_if_fail(ST_IS_CATEGORY_BAG(bag), NULL);

  return bag->priv->task_thread;
}
