(****************************************************************************)
(*                           the diy toolsuite                              *)
(*                                                                          *)
(* Jade Alglave, University College London, UK.                             *)
(* Luc Maranget, INRIA Paris-Rocquencourt, France.                          *)
(*                                                                          *)
(* Copyright 2014-present Institut National de Recherche en Informatique et *)
(* en Automatique and the authors. All rights reserved.                     *)
(*                                                                          *)
(* This software is governed by the CeCILL-B license under French law and   *)
(* abiding by the rules of distribution of free software. You can use,      *)
(* modify and/ or redistribute the software under the terms of the CeCILL-B *)
(* license as circulated by CEA, CNRS and INRIA at the following URL        *)
(* "http://www.cecill.info". We also give a copy in LICENSE.txt.            *)
(****************************************************************************)

module type Config = sig
  val verbose : int
  val cond : Config.cond
  val optcond : bool
  val coherence_decreasing : bool
  val hexa : bool
end

module Make : functor (O:Config) -> functor (C:ArchRun.S) ->
  sig
    type fenv = (C.A.location * IntSet.t) list
    type eventmap = C.A.location C.C.EventMap.t

(* Add an observation to fenv *)
    val add_final_v :
        Code.proc -> C.A.arch_reg -> IntSet.t -> fenv -> fenv
    val add_final :
        Code.proc -> C.A.arch_reg option -> C.C.node ->
          eventmap * fenv -> eventmap * fenv

    type final

    val check : fenv -> final
    val observe : fenv -> final
    val run : C.C.event list list -> C.A.location C.C.EventMap.t -> final

    val dump_final : out_channel ->  final -> unit

  end = functor (O:Config) -> functor (C:ArchRun.S) ->
  struct
    type fenv = (C.A.location * IntSet.t) list
    type eventmap = C.A.location C.C.EventMap.t

    let show_in_cond =
      if O.optcond then
        let valid_edge m =
          let e = m.C.C.edge in
          let open C.E in
          match e.C.E.edge with
          | Rf _ | Fr _ | Ws _ | Hat
          | Back _|Leave _ -> true
          | Po _ | Fenced _ | Dp _|Rmw|Insert _|Node _ -> false
          | Id -> assert false in
        (fun n ->
          let p = C.C.find_non_pseudo_prev n.C.C.prev in
          valid_edge p || valid_edge n)
      else
        (fun _ -> true)

    let add_final_v p r v finals = (C.A.of_reg p r,v)::finals

    let prev_value =
      if O.coherence_decreasing then fun v -> v+1
      else fun v -> v-1

    let add_final p o n finals = match o with
    | Some r ->
        let m,fs = finals in
        let evt = n.C.C.evt in
        let v = match evt.C.C.dir with
        | Some Code.R -> Some evt.C.C.v
        | Some Code.W -> Some (prev_value evt.C.C.v)
        | None -> None in
        if show_in_cond n then match v with
        | Some v ->
            C.C.EventMap.add n.C.C.evt (C.A.of_reg p r) m,
            add_final_v p r (IntSet.singleton v) fs
        | None -> finals
        else finals
    | None -> finals


    type final =
      | Exists of fenv 
      | Forall of (C.A.location * Code.v) list list
      | Locations of C.A.location list

    module Run = Run_gen.Make(O)(C)

    let check f = Exists f
    let observe f = Locations (List.map fst f)
    let run evts m = Forall (Run.run evts m)

(* Dumping *)
    open Printf

    let dump_atom =
      if O.hexa then
        fun r v -> sprintf "%s=0x%x" (C.A.pp_location r) v
      else
        fun r v -> sprintf "%s=%i" (C.A.pp_location r) v

    let dump_state fs =
      String.concat " /\\ "
        (List.map
           (fun (r,vs) ->
             match IntSet.as_singleton vs with
             | Some v ->
                 dump_atom r v
             | None ->
                 let pp =
                   IntSet.pp_str " \\/ "
                     (fun v -> dump_atom r v)
                     vs in
                 sprintf "(%s)" pp)
           fs)

    let dump_final chan f = match f with
    | Exists fs ->
        fprintf chan "%sexists\n" (if !Config.neg then "~" else "") ;
        fprintf chan "(%s)\n" (dump_state fs)
    | Forall ffs ->
        fprintf chan "forall\n" ;
        fprintf chan "%s\n" (Run.dump_cond ffs)
    | Locations locs ->
        fprintf chan "locations [%s]\n"
          (String.concat ""
             (List.map (fun loc -> sprintf "%s;" (C.A.pp_location loc)) locs))

  end

