#! /bin/bash

# /usr/local/sbin/twsetup.sh
# https://sourceforge.net/projects/mktwpol

[[ "${EUID}" != "0" ]]		&& echo " Got root?"		&& exit
[[ ! "${BASH}" =~ "bash" ]]	&& echo " ${0##*/} needs bash"	&& exit

VERSION=1.0.1		# 01 APR 2017

# Copyright (C) 2013-2017 Chuck Seyboldt <c.cboldt@gmail.com>
# This work is licensed under a Creative Commons Attribution-ShareAlike 3.0 Unported License
# http://creativecommons.org/licenses/by-sa/3.0/
#
# make_key_files and sign_twcfg_file subroutines are adaptations of
# Tripwire(R) 2.4 Open Source install script, found at
# install/install.sh and install/install.cfg, in tripwire-2.4.2.2-src tarball

# A script to configure tripwire and set-up a tripwire database
# Integrates tightly with a companion script, mktwpol.sh
#  - mktwpol.sh generates plain-text tripwire policy from
#    the system's database of installed packages

# ======================================================================
# A small number of initial variable assignments.  Leave them alone.

# If CLOBBER==true, overwrite key and (signed) tw.cfg files
# If CLOBBER==false, use existing key files and make backup of tw.cfg file
CLOBBER="false"

# Local and Site Passphrase variables (from tripwire's install.sh)
# Danger! Danger! Will Robinson!  (See README and mktwpol.cfg for remarks)
LOCAL_PASSPHRASE=""
SITE_PASSPHRASE=""

# End of initial script variable assignments
# There is REALLY nothing adjustable below this line, other than bugs
# ======================================================================
# The default values that appear in the script can be changed using
# twsetup.sh command line switches or settings in mktwpol.cfg

# TW_CFG is a key variable.  Other variables are established from
# knowing the name and location of tripwire's configuration file.
# If you want to configure into a different directory /etc/tw-sandbox,
# use the "-c" switch (not settable from mktwpol.cfg file)
# Run:	`twsetup.sh -c /etc/tw-sandbox`			assigned directory
#  or	`twsetup.sh -c /etc/tw-sandbox/tw.cfg`		default cfg name
#  or	`twsetup.sh -c /etc/tw-sandbox/tw-sandbox.cfg`	assigned cfg name

TW_CFG=${TW_CFG:=/etc/tripwire/tw.cfg}

# MKTWPOL_CMD_LINE is a command that will produce plain-text tripwire
# policy on STDOUT.  If you want to use a different command to generate
# plain-text tripwire policy, use the "-p" switch
# Run:	`twsetup.sh -p "mktwpol-generic.sh"` (use quotes to escape spaces)
#
# Note: FORCE_PRINT=Y passed to force mktwpol.sh to print to STDOUT

MKTWPOL_CMD_LINE=${MKTWPOL_CMD_LINE:="mktwpol.sh"}

# These 5 variables are read from twcfg.txt (or tw.cfg, if it exists)
# See config_twsetup subroutine, just below
#
# Variable		Typical Assignment in twtxt.cfg
# ============		================================================
# SITEKEYFILE		${TW_CFG_DIR}/site.key
# LOCALKEYFILE		${TW_CFG_DIR}/$(HOSTNAME)-local.key
# POLFILE		${TW_CFG_DIR}/tw.pol
# DBFILE		/var/lib/tripwire/$(HOSTNAME).twd
# REPORTFILE		/var/lib/tripwire/report/$(HOSTNAME)-$(DATE).twr

# =======   Start of Subroutines   =======

# "config_twsetup" assigns variables values from several sources
# "introduce_twsetup" describes what the script intends to perform
# "make_txt_cfg" makes a plain-text TRIPWIRE configuration file
# "confirm_twcfg" asks to confirm config file, and provides an out
# "check_db_report_dirs" checks for and makes working directories
# "key_file_maker" adapted from tripwire install.sh, makes site.key and local.key
# "sign_twcfg_file" adapted from tripwire install.sh, makes tw.cfg file
# "make_plain_text_policy" introduces and calls `mktwpol.sh`
# "encrypt_policy" calls twadmin to encrypt the plain-text policy file
# "scan_filesystem" calls tripwire to create its database
# "make_cronjob" makes new or inspects exisiting tripwire cronjob
# "outro_twsetup" claims success and presents opportunity to prove it

# ==========================================================================
# "config_twsetup" assigns variables values from several sources
# ==========================================================================

config_twsetup ()
{
# If twsetup.sh is called with TW_CFG set to an existing directory,
# or if the "basename" part of TW_CFG has no dot, as though being a
# config filename, then treat as though TW_CFG is a directory name,
# and use the "tw.cfg" and "twcfg.txt" tripwire naming conventions
# for the tripwire configuration files.

if [ -d ${TW_CFG} -o "${TW_CFG}" == "${TW_CFG%.*}" ]; then
  TW_CFG_DIR=${TW_CFG}
  TW_CFG=${TW_CFG_DIR}/tw.cfg
fi

# If TW_CFG has no slash "/" in it, treat it as a config file name
# and use /etc/tripwire default installation directory

if [ "${TW_CFG}" == "${TW_CFG%/*}" ]; then
  TW_CFG=/etc/tripwire/${TW_CFG}
fi

# TW_TXT_CFG is hard-coded as "twcfg.txt".  No command line switch provides
# for deviation from the "twcfg.txt" basename

			 # bash tip: `basename` command == ${TW_CFG##*/}
TW_CFG_DIR=${TW_CFG%/*}  # bash shell equivalent of the `dirname` command
TW_TXT_CFG=$TW_CFG_DIR/twcfg.txt	# hard-coded use of "twcfg.txt"

# TW_CFG, TW_CFG_DIR, and TW_TXT_CFG are frozen at this point
# -----------------------------------------------------------
# This is protection from attempted mischief in mktwpol.cfg

readonly TW_CFG TW_CFG_DIR TW_TXT_CFG

# Read twsetup.sh configuration from a mktwpol.cfg if it exists

CONFIG_FILE=${CONFIG_FILE:="`ls -t ${TW_CFG_DIR}/mktwpol*.cfg /etc/{mktwpol,tripwire}/mktwpol*.cfg /root/mktwpol.cfg 2> /dev/null | head -1`"}

if [ -r ${CONFIG_FILE} ]; then
  if [ "`stat -c %A ${CONFIG_FILE}`" != "-rw-------" ]; then
    echo " Exiting because of config file permissions."
    echo " If you don't move, remove or rename it ..."
    echo " Run \`chmod 600 ${CONFIG_FILE}\` and try again."
    exit 8
  elif [ "`stat -c %U ${CONFIG_FILE}`" != "root" ]; then
    echo " Exiting because config file owner is not root."
    echo " If you don't move, remove or rename it ..."
    echo " Run \`chown root ${CONFIG_FILE}\` and try again."
    exit 8
  fi
  source ${CONFIG_FILE}
fi

# NONSTD_TW_CFG is used to disambiguate some tripwire commands, if TW_CFG
# is something other than tripwire's default /etc/tripwire/tw.cfg

[ "$TW_CFG" != "/etc/tripwire/tw.cfg" ] && NONSTD_TW_CFG=" -c ${TW_CFG}"

# Make tripwire config directory if none exists

if [ ! -d ${TW_CFG_DIR} ]; then
  echo
  echo " Creating Tripwire configuration directory : ${TW_CFG_DIR}"
  mkdir -p ${TW_CFG_DIR} ; chmod 750 ${TW_CFG_DIR}
fi

# TW_BASE is one word.  The default for that word is "tripwire"
# TW_BASE is used to name database and report directories, and
# to name the cronjob.
TW_BASE=${TW_CFG_DIR##*/}

# Name the plain-text policy file
# This is settable via mktwpol.cfg file

TXT_POLFILE=${TXT_POLFILE:=${TW_CFG_DIR}/twpol-`date +%y%m%d-%H%M`.txt}
EXISTING_TXT_POLFILE="`ls -t ${TW_CFG_DIR}/twpol*txt /etc/{mktwpol,tripwire}/twpol*txt /root/twpol*txt /var/lib/mktwpol/twpol*txt 2> /dev/null | head -1`"

# If policy generator is bypassed, test for a plain-text policy file.
# Sneak in substitutions for "YOUR_HOSTNAME" and "^HOSTNAME=" strings.

if [[ "${BYPASS_GENERATOR^}" =~ "Y" ]]; then
  grep -m1 rulename ${EXISTING_TXT_POLFILE} >& /dev/null
  if [ "$?" != "0" ]; then
    echo
    echo " Nice try.  Caught you!  Provide a plain-text policy."
    echo " Its name must be \"twpol*txt\" and it must have a defined rulename."
    echo " Exiting, stage left."
    echo
    exit 13
  fi
  sed -i -e s/YOUR_HOSTNAME/${HOSTNAME}/ \
	 -e s/^HOSTNAME=/#HOSTNAME=/ ${EXISTING_TXT_POLFILE}
  echo
  echo " Bypassing plain-text policy generator."
  echo " I will use \`cat ${EXISTING_TXT_POLFILE}\` as a pseudo-generator."
  MKTWPOL_CMD_LINE="cat ${EXISTING_TXT_POLFILE}"
  echo
  printf " Press a key when you are ready to continue ... "
  read -n 1 READY_TO_CONTINUE
fi

# Strip off everything following the first space in MKTWPOL_CMD_LINE
MKTWPOL_CMD=${MKTWPOL_CMD_LINE%% *}

# Test for MKTWPOL_CMD, even though twsetup.sh and mktwpol.sh are
# shipped as a unit, and are supposed to be installed as a unit.
# If no policy generator, test for a plain-text policy file.  If a
# plain-text policy file is found, `cat` the most recent one.

hash ${MKTWPOL_CMD} 2> /dev/null
if [ ! $? = 0 ]; then
  if [ "${EXISTING_TXT_POLFILE}" == "" ]; then
    echo
    echo " Oooops!!!"
    echo " I can't find either a plain-text policy generator, or an existing"
    echo " plain-text policy file!  The default plain-text policy file is twpol.txt"
    echo
    echo " The script absolutely needs either a plain-text policy generator"
    echo " or an existing plain-text policy file."
    echo
    echo " Examples of plain-text policy files can be found in the tripwire"
    echo " tarball \"policy/\" folder."
    echo
    echo " Exiting.  Goodbye."
    echo
    exit 10
  else
    echo
    echo " NOTICE!"
    echo " ======="
    echo " Missing the requested tripwire policy generator program."
    echo " \`${MKTWPOL_CMD}\` is not available for making plain-text policy."
    echo " I will use \`cat ${EXISTING_TXT_POLFILE}\` instead."
    MKTWPOL_CMD_LINE="cat ${EXISTING_TXT_POLFILE}"
    MKTWPOL_CMD=cat
    echo
    printf " Press a key when you are ready to continue ... "
    read -n 1 READY_TO_CONTINUE
  fi
# Test mktwpol.sh
# Must contain FORCE_PRINT
elif [ "${MKTWPOL_CMD}" == "mktwpol.sh" ]; then
  if [ -z "$(grep -m1 -e FORCE_PRINT `command -v ${MKTWPOL_CMD}`)" ]; then
    echo " `command -v ${MKTWPOL_CMD}` is not compatible with $0."
    echo " Upgrade your version of ${MKTWPOL_CMD}"
    echo
    exit 1
  fi
fi

# avoid `cat file > file` error if EXISTING_TXT_POLFILE == TXT_POLFILE
# EXISTING_TXT_POLFILE is preserved
# Date-coded TXT_POLFILE is deleted after `cat` command

if [ "${MKTWPOL_CMD}" == "cat" ] && \
   [ "${EXISTING_TXT_POLFILE}" == "${TXT_POLFILE}" ]; then
   TXT_POLFILE=${TW_CFG_DIR}/twpol-`date +%y%m%d-%H%M`.txt
fi

# ----- Reading the TRIPWIRE configuation file -----

# $TW_CFG_DIR/twcfg.txt or $TW_CFG_DIR/tw.cfg must exist
# If neither exists, the script writes a $TW_CFG_DIR/twcfg.txt

# Test for encrypted TW_CFG (TW_CFG_DIR/tw.cfg) first.
# If TW_CFG exists, tripwire has already been configured.

if [ -r ${TW_CFG} -a "${AUTO_UPDATE}" != "Yes" ]; then
  echo " It appears tripwire has already been configured."
  if [ ${TW_CFG} -ot ${TW_TXT_CFG} ]; then
    echo " But ${TW_TXT_CFG} has been changed since ${TW_CFG} was made."
    echo " The script will reconfigure tripwire using ${TW_TXT_CFG}"
  else
    echo " There is probably no need to run this script."
    if [[ "${MKTWPOL_CMD_LINE}" =~ "mktwpol" ]]; then
      echo
      echo " If you want to update the tripwire policy file and database,"
      echo " Press \"Y\" to run \`${MKTWPOL_CMD}\`"
    fi
    echo
    printf " Update tripwire database instead of initial setup? [y/N]: "
    read -n 1 YES_CONTINUE
    echo
    if [ "${YES_CONTINUE^}" == "Y" ]; then
      AUTO_UPDATE=Yes
    fi
  fi
fi

if [ ! -r ${TW_TXT_CFG} -a ! -r ${TW_CFG} ]; then
  echo "
 ${0##*/} and tripwire depend on finding a tripwire configuration file
	${TW_TXT_CFG}   is expected before tripwire is set-up
	${TW_CFG}      is the same information, \"signed\" during set-up

 ${0##*/} found neither."
  make_txt_cfg
fi

# The "if TW_TXT_CFG elif TW_CFG" subroutine below stands separate from the
# the "if TW_CFG elif TW_TXT_CFG" logic above in order to make the source
# and contents of tripwire variables consistent between printing the intro
# and printing the tripwire variables for inspection and agreement.

if [ -r ${TW_TXT_CFG} ]; then
  WHICH_TW_CFG="plain-text ${TW_TXT_CFG}"
  tmp_array=(`cat ${TW_TXT_CFG}`)
elif [ -r ${TW_CFG} ]; then
  WHICH_TW_CFG="encrypted ${TW_CFG}"
  tmp_array=(`twadmin --print-cfgfile -c ${TW_CFG}`)
fi

# `z+=2` incrementing shaves a few thousandths of a second
# but risks skipping over the required variable names

for (( z = 0 ; z < ${#tmp_array[@]} ; z++ ))
do
  case ${tmp_array[@]:$z:1} in
    POLFILE | DBFILE | REPORTFILE | SITEKEYFILE | LOCALKEYFILE  )
      assignment=${tmp_array[@]:$((z+1)):1}
      [ "${assignment:0:1}" == "=" ] && \
        export ${tmp_array[@]:$z:1}${assignment}
  ;;
  esac
done

REPORTDIR=${REPORTFILE%/*}  # bash shell equivalent of the `dirname` command
DBDIR=${DBFILE%/*}

# Using bash built-in string substitution to deal with tripwire convention
# of using parentheses instead of curley braces to denote a variable

DBFILE=${DBFILE/\$\(HOSTNAME\)/$HOSTNAME}
LOCALKEYFILE=${LOCALKEYFILE/\$\(HOSTNAME\)/$HOSTNAME}
}

# ================================================================
# "introduce_twsetup" describes what the script intends to perform
# ================================================================

introduce_twsetup ()
{
echo "
 Both \`tripwire\` and \`twadmin\` exist.
 The ${WHICH_TW_CFG} config file has been read.

 This script will undertake these actions:

 1. Show the contents of ${WHICH_TW_CFG} for review and approval
     * You can stop the script to edit plain-text ${TW_TXT_CFG}
     * If necessary, the script will make database and report directories

 2. Create a site key file, currently configured as ${SITEKEYFILE}
    Command: \`twadmin --generate-keys -S ${SITEKEYFILE}\`
     * You will be prompted to establish a \"Site Passphrase\"

 3. Create a local key file, currently configured as ${LOCALKEYFILE}
    Command: \`twadmin --generate-keys -L ${LOCALKEYFILE}\`
     * You will be prompted to establish a \"Local Passphrase\"

 4. Encrypt ${TW_TXT_CFG} to ${TW_CFG}
    Command: \`twadmin --create-cfgfile${NONSTD_TW_CFG} \\
	     -S ${SITEKEYFILE} ${TW_TXT_CFG}\`
     * You will prompted to input the \"Site Passphrase\"

 5. Generate a plain-text tripwire policy file (generically, twpol.txt).
    The plain-text tripwire policy file establishes which files will be
    inspected by tripwire, and what file changes tripwire will report.
    This script will date-code the plain-text policy file.
    Command: \`${MKTWPOL_CMD_LINE} > $TXT_POLFILE\`

 6. Encrypt $TXT_POLFILE to ${POLFILE}
    Command: \`twadmin --create-polfile${NONSTD_TW_CFG} \\
	     $TXT_POLFILE\`
     * You will prompted to input the \"Site Passphrase\"

 7. Create an encrypted tripwire database, ${DBFILE}
    Command: \`tripwire --init${NONSTD_TW_CFG}\`
     * You will be prompted to input the \"Local Passphrase\"
"
printf " Press a key when you are ready to continue ... "
read -n 1 READY_TO_CONTINUE
echo
}

# ======================================================================
# "make_txt_cfg" makes a plain-text TRIPWIRE configuration file
# ======================================================================

make_txt_cfg ()
{
DB_ROOT=${DB_ROOT:=/var/lib}
TMP_DIR=${TMP_DIR:=/tmp}
echo " Making a plain-text $TW_TXT_CFG file now."
echo "# Changes to this file are NOT effective, until running the command:
# twadmin --create-cfgfile${NONSTD_TW_CFG} -S ${TW_CFG_DIR}/site.key ${TW_TXT_CFG}

POLFILE			=${TW_CFG_DIR}/tw.pol
DBFILE			=${DB_ROOT}/$TW_BASE/\$(HOSTNAME).twd
REPORTFILE		=${DB_ROOT}/$TW_BASE/report/\$(HOSTNAME)-\$(DATE).twr
SITEKEYFILE		=${TW_CFG_DIR}/site.key
LOCALKEYFILE		=${TW_CFG_DIR}/\$(HOSTNAME)-local.key

EDITOR			=${EDITOR:=/bin/nano}
TEMPDIRECTORY		=$TMP_DIR
# GLOBALEMAIL		=
LATEPROMPTING		=false
LOOSEDIRECTORYCHECKING	=false
SYSLOGREPORTING		=true
REPORTLEVEL		=3
EMAILREPORTLEVEL	=3
MAILNOVIOLATIONS	=true

# MAILMETHOD		=SMTP
# SMTPHOST		=mail.domain.com
# SMTPPORT		=25
" > ${TW_TXT_CFG}

hash sendmail 2> /dev/null
[ $? != 0 ] && HASHMARK="# "
echo "${HASHMARK}MAILMETHOD		=SENDMAIL
${HASHMARK}MAILPROGRAM		=`command -v sendmail` -oi -t
" >> ${TW_TXT_CFG}
chmod 600 ${TW_TXT_CFG}
printf " Press a key when you are ready to continue ... "
read -n 1 READY_TO_CONTINUE
}

# =================================================================
# "confirm_twcfg" asks to confirm config file, and provides an out
# =================================================================

# The if/elif/else order must be the same as the routine in config_twsetup
# which establishes $WHICH_TW_CFG, or else the contents of $TW_CFG might
# be claimed as the contents of $TW_TXT_CFG, or vice versa

confirm_twcfg ()
{
echo " Step 1 - Showing contents of ${WHICH_TW_CFG} ..."
echo
if [ -r ${TW_TXT_CFG} ]; then
  cat ${TW_TXT_CFG}
elif [ -r ${TW_CFG} ]; then
  twadmin --print-cfgfile -c ${TW_CFG}
fi
echo
echo " The use of parentheses to denote \$(HOSTNAME) and \$(DATE) is normal."
echo " \`man twconfig\` to learn about the function of the tripwire variables."
echo
printf " Are ALL of those settings suitable? [y/N]: "
read -n 1 YES_THEY_ARE
echo
[ ! -f ${TW_TXT_CFG} ] && twadmin -m f -c ${TW_CFG} > ${TW_TXT_CFG}
if [ "${YES_THEY_ARE^}" != "Y" ]; then
  echo " OK, edit ${TW_TXT_CFG} and try again."
  echo " Goodbye."
  echo
  exit
fi
}

# ================================================================
# "check_db_report_dirs" checks for and makes working directories
# ================================================================

check_db_report_dirs ()
{
if [ ! -d ${DBDIR} ]; then
  echo
  echo " Creating Tripwire database directory : ${DBDIR}"
  mkdir -p ${DBDIR} ; chmod 700 ${DBDIR}
fi

if [ ! -d ${REPORTDIR} ]; then
  echo " Creating Tripwire report directory   : ${REPORTDIR}"
  mkdir -p ${REPORTDIR} ; chmod 700 ${REPORTDIR}
fi
}

# ===============================================================================
# "key_file_maker" adapted from tripwire install.sh, makes site.key and local.key
# ===============================================================================

#  Indirection used to get to the contents of $SITEKEYFILE, etc.
#  "i" is either "site" or "local"; "I" is either "SITE" or "LOCAL"
#  ${!IKEYFILE}     is ${SITEKEYFILE},	   then ${LOCALKEYFILE}
#  ${!I_PASSPHRASE} is ${SITE_PASSPHRASE}, then ${LOCAL_PASSPHRASE}

key_file_maker ()
{
(( c++ ))			# Increment "Rule No." counter
I=${i^^}			# capitalize, e.g., $i=site, $I=SITE
IKEYFILE=${I}KEYFILE		# $IKEYFILE contains a variable name
I_PASSPHRASE=${I}_PASSPHRASE	# $I_PASSPHRASE contains a variable name

echo
echo " Step $c - Creating $i key file: ${!IKEYFILE} ..."

# If CLOBBER is true, and the key file already exists, remove it.
# Otherwise twadmin will prompt with an "are you sure?" message.

if [ "$CLOBBER" = "true" ] && [ -f "${!IKEYFILE}" ] ; then
	rm -f "${!IKEYFILE}"
fi

if [ -f "${!IKEYFILE}" ] && [ "$CLOBBER" = "false" ] ; then
	echo " The $i key file \"${!IKEYFILE}\""
	echo ' exists and will not be overwritten.'
else
	cmdargs="--generate-keys --${i}-keyfile \"${!IKEYFILE}\""
	if [ -n "${!I_PASSPHRASE}${NULL_PASSPHRASE}" ] ; then
		cmdargs="$cmdargs --${i}-passphrase \"${!I_PASSPHRASE}\""
	fi
	eval "twadmin $cmdargs"
	if [ $? -ne 0 ] ; then
		echo " Error: $i key generation failed"
		exit 1
	else chmod 600 "${!IKEYFILE}"
	fi
fi
}

# Call key_file_maker to make site key, then to make local key

make_key_files ()
{
c=1				# Initialize counter for "Rule No."
for i in site local ; do
  key_file_maker		# Call the routine that does the work
done
}

# =====================================================================
# "sign_twcfg_file" adapted from tripwire install.sh, makes tw.cfg file
# =====================================================================

sign_twcfg_file ()
{
echo
echo " Step 4 - Making encrypted configuration file: ${TW_CFG} ..."
echo

##-------------------------------------------------------
## If noclobber, then move existing tw.cfg config file.
##-------------------------------------------------------

if [ "$CLOBBER" = "false" ] && [ -s "$TW_CFG" ] ; then
	backup="${TW_CFG}.$$.bak"
	echo " Backing up $TW_CFG"
	echo "         to $backup"
	`mv "$TW_CFG" "$backup"`
	if [ $? -ne 0 ] ; then
		echo " Error: backup of configuration file failed."
		exit 1
	fi
fi

##-------------------------------------------------------
## Create the encrypted tripwire configuration file.
##-------------------------------------------------------

cmdargs="--create-cfgfile --site-keyfile \"$SITEKEYFILE\""
cmdargs="$cmdargs --cfgfile \"$TW_CFG\""
if [ -n "${SITE_PASSPHRASE}${NULL_PASSPHRASE}" ] ; then
	cmdargs="$cmdargs --site-passphrase \"$SITE_PASSPHRASE\""
fi

eval "twadmin $cmdargs \"$TW_TXT_CFG\""
if [ $? -ne 0 ] ; then
	echo " Error: failed to make tripwire configuration file ${TW_CFG}"
	exit 1
fi
chmod 600 "$TW_CFG"

if [[ "${AUTO_RM^}" =~ "Y" ]]; then
  rm $TW_TXT_CFG
  echo "Deleted plain-text configuration file: $TW_TXT_CFG"
  echo "It can be seen with: \`twadmin --print-cfgfile${NONSTD_TW_CFG}\`"
else
  echo "
 Feel free to delete $TW_TXT_CFG
 The contents of $TW_TXT_CFG can be retrieved
 Run: \`twadmin --print-cfgfile${NONSTD_TW_CFG}\`"
fi
}

# ==========================================================
# "make_plain_text_policy" introduces and calls `mktwpol.sh`
# ==========================================================

make_plain_text_policy ()
{
# export of TW_CFG passes non-standard TW_CFG to mktwpol.sh
# Less confusing than showing and passing "-c ${NONSTD_TW_CFG}" parm
# $TW_CFG only affects mktwpol.sh, does not interfere with any possible
# third party policy generator, and does not interfere with `cat`

export TW_CFG
export TXT_POLFILE
export RULES_FILE
FORCE_PRINT="Yes"
export FORCE_PRINT
echo
if [ "${MKTWPOL_CMD}" == "cat" ]; then
  echo " Step 5 - Copying $EXISTING_TXT_POLFILE to: $TXT_POLFILE ..."
else
  echo " Step 5 - Generating plain-text policy file  ..."
fi
trap trap_make_plain_text_policy SIGHUP SIGINT SIGTERM
echo "  # $0 -> \`${MKTWPOL_CMD_LINE}\` > $TXT_POLFILE" > $TXT_POLFILE
${MKTWPOL_CMD_LINE} >> $TXT_POLFILE
[ "$?" != "0" ] && trap_make_plain_text_policy
chmod 600 $TXT_POLFILE
trap "echo; echo ${0##*/} interrupted; echo; exit" SIGHUP SIGINT SIGTERM
}

trap_make_plain_text_policy ()
{
echo
echo
echo " Oooops"
echo
echo " $MKTWPOL_CMD was interrupted before it completed generating the"
echo " plain-text tripwire policy file."
echo
echo " $TXT_POLFILE may be empty, or incomplete but available for"
echo " review and editing."
echo
echo " Goodbye."
echo
exit 5
}

# ====================================================================
# "encrypt_policy" calls twadmin to encrypt the plain-text policy file
# ====================================================================

encrypt_policy ()
{
echo
echo " Step 6 - Making encrypted policy file: ${POLFILE} ..."
if [ -n "${SITE_PASSPHRASE}${NULL_PASSPHRASE}" ]; then
  twadmin --create-polfile ${NONSTD_TW_CFG} \
          --site-passphrase "${SITE_PASSPHRASE}" $TXT_POLFILE
else
  twadmin --create-polfile ${NONSTD_TW_CFG} $TXT_POLFILE
fi

if [ "$?" != 0 ]; then
  echo
  echo " There is something wrong with plain-text policy file $TXT_POLFILE"
  echo " Tripwire demands it to be fixed before encrypting it to ${POLFILE}"
  echo " Usually, this error follows multiple or overlapping policy definitions"
  echo " for one or more directory or filenames."
  echo
  if [ "$MKTWPOL_CMD" == "cat" ]; then
    rm $TXT_POLFILE
    echo " No plain-text generator involved, edit ${EXISTING_TXT_POLFILE}"
    echo " Exiting.  Good luck!"
    echo
    exit 11
  else
    echo " The plain-text policy generator created this error."
    echo " To bypass it, use \`${0##/} -b\`"
    echo " Exiting.  Good luck!"
    echo
    exit 12
  fi
else
  chmod 600 ${POLFILE}
fi


if [ "$MKTWPOL_CMD" == "cat" ]; then
  echo
  echo " Deleting $TXT_POLFILE"
  echo " It was merely a copy of ${EXISTING_TXT_POLFILE}"
  rm $TXT_POLFILE
elif [[ "${AUTO_RM^}" =~ "Y" ]] || [ -n "${NULL_PASSPHRASE}" ]; then
  rm $TXT_POLFILE
  echo "Deleted plain-text policy file: $TXT_POLFILE"
  echo "It can be seen with: \`twadmin --print-polfile${NONSTD_TW_CFG}\`"
else
 echo "
 Feel free to delete the plain-text policy file, $TXT_POLFILE
 The contents of the plain-text policy file can be retrieved ...
 Run: \`twadmin --print-polfile${NONSTD_TW_CFG}\`"
fi
}

# =======================================================
# "scan_filesystem" calls tripwire to create its database
# =======================================================

# If the tripwire policy includes files not existing on the
# system, then `tripwire --init` will throw error messages,
# but will exit with errorlevel 0.

scan_filesystem ()
{
# Note bash method for redirecting STDERR to a variable
# Must act on the variable within the parentheses
# command 2> >(readarray variable)
# command 2> >(variable=$(cat))

ERROR_FILE=${TW_CFG_DIR}/`date +%y%m%d-%H%M`-TW-ERRORS.txt
echo
echo " Tripwire may take a few minutes to scan the filesystem."
echo
echo " Step 7 - Creating the tripwire database: ${DBFILE} ..."
if [ -n "${LOCAL_PASSPHRASE}${NULL_PASSPHRASE}" ]; then
  tripwire --init $NONSTD_TW_CFG --local-passphrase "${LOCAL_PASSPHRASE}" 2> \
  >(IFS=$'\n'; readarray tw_err; [ -n "$tw_err" ] && \
  for i in ${tw_err[@]}; do echo $i; done > $ERROR_FILE)
else
  tripwire --init $NONSTD_TW_CFG 2> \
  >(IFS=$'\n'; readarray tw_err; [ -n "$tw_err" ] && \
  for i in ${tw_err[@]}; do echo $i; done > $ERROR_FILE)
fi

if [ -s $ERROR_FILE ]; then
  echo
  echo " The database was created, but there are some non-fatal errors."
  echo " A record of them is in: $ERROR_FILE"
  echo
  if [ "${AUTO_UPDATE}" != "Yes" ] ;then
    printf " Press a key when you are ready to continue ... "
    read -n 1 READY_TO_CONTINUE
  fi
fi
}

# ===============================================================
# "make_cronjob" makes new or inspects exisiting tripwire cronjob
# ===============================================================

make_cronjob ()
{
TRIPWIRE_CRON=""
TRIPWIRE_CRON=${TRIPWIRE_CRON:=`ls /etc/cron.*/${TW_BASE}* 2> /dev/null`}
TRIPWIRE_CRON=${TRIPWIRE_CRON:="No tripwire cronjob found"}

# Subroutine to write a cronjob from scratch

if [ "${TRIPWIRE_CRON}" == "No tripwire cronjob found" ] ; then
  echo
  printf " ${TRIPWIRE_CRON}.  Write /etc/cron.daily/${TW_BASE} now? [y/N] "
  read -n 1 -t 60 WRITE_CRONJOB
  if [ "${WRITE_CRONJOB^}" == "Y" ]; then
    TRIPWIRE_CRON=/etc/cron.daily/${TW_BASE}
    echo "#!/bin/bash

# ${TRIPWIRE_CRON}
# =====================================================
# Added by $0 $VERSION
# `date`
# =====================================================
#
# To create report files that are encrypted with the local passphrase,
# the cronjob command line would have to tell tripwire to encrypt the
# report, and also give the passphrase.
#
# `command -v tripwire` --check --quiet${NONSTD_TW_CFG} -E -P \"secretpassphrase\"
#
# An alternative to exposing \"secretpassphrase\" in the cronjob
# is to \`chmod 700 ${REPORTDIR}\`

if [ ! -e ${DBFILE} ] ; then
  echo \"****    Error: Tripwire database for \`uname -n\` not found.    ****\"
  echo \"****         Check tripwire.txt file for instructions           ****\"
elif [ -f ${TW_CFG} ] ; then
  echo
  `command -v tripwire` --check --quiet${NONSTD_TW_CFG}
fi
" > ${TRIPWIRE_CRON}
  chmod 750 ${TRIPWIRE_CRON}
  echo
  fi
fi

# Subroutine to append tmpwatch to existing cronjob
# Tests for existence of tmpwatch executable

hash tmpwatch 2> /dev/null ; [ ! $? = 0 ] && return
[ ! -s "${TRIPWIRE_CRON}" ] && return
[ ! -z "`grep -i -e tmpwatch.*${REPORTDIR} ${TRIPWIRE_CRON}`" ] && return

echo "
 This system has a cronjob that runs tripwire on a
 scheduled basis.  The cronjob file is: ${TRIPWIRE_CRON}

 ${TRIPWIRE_CRON} does nothing to remove old tripwire reports.
 You might want to add a tmpwatch subroutine to it.

 To automatically remove 336 hour old (two weeks old) tripwire reports,
 the tmpwatch subroutine would look like this:

	if [ -d ${REPORTDIR} ]; then
	   `command -v tmpwatch` --ctime 336 ${REPORTDIR}
	fi
"
printf " Append that routine to ${TRIPWIRE_CRON}? [y/N] "
read -n 1 -t 60 WRITE_TMPWATCH
echo
if [ "${WRITE_TMPWATCH^}" == "Y" ]; then
  echo "
# ========================================================
# Added by $0 $VERSION
# Removes tripwire reports that are older than ctime hours
# `date`
# ========================================================

if [ -d ${REPORTDIR} ]; then
   `command -v tmpwatch` --ctime 336 ${REPORTDIR}
fi
" >> ${TRIPWIRE_CRON}
fi
}

# ===================================================================
# "outro_twsetup" claims success and presents opportunity to prove it
# ===================================================================

outro_twsetup ()
{
echo "

 Tripwire set-up is complete!
 ============================

 The task ${0##*/} is designed to perform is done, fini, over.
 From now on, when the tripwire database needs attention, run either
 \`tripwire --update\` or \`mktwpol.sh -u\`

 Unless \`tripwire --check\` produces a \"Warning: File system error\"
 message, there is no need to rebuild the policy file or tripwire database.
 Use the \`tripwire --update\` facility instead of mktwpol.sh
 Run: \`tripwire --update -r \$(ls -t ${REPORTDIR}/* | head -1)${NONSTD_TW_CFG}\`

 Tip: Add these aliases to /root/.bashrc
      alias last.tw.report='echo \$(ls -t ${REPORTDIR}/* | head -1)'
      alias tw.report='twprint  --print-report -r \`last.tw.report\`'
      alias tw.update='tripwire --update -r \`last.tw.report${NONSTD_TW_CFG}\`'

 When \`tripwire --check\` produces \"Warning: File system error\" messages,
      rebuild the tripwire policy and database.  Please ...
 Run: \`mktwpol.sh -u${NONSTD_TW_CFG}\` instead of the ${0##*/} script.

 Tip: To view the tripwire config after deleting \"$TW_TXT_CFG\"
 Run: \`twadmin --print-cfgfile${NONSTD_TW_CFG}\`

 Tip: To view the tripwire policy after deleting the plain-text policy file
 Run: \`twadmin --print-polfile${NONSTD_TW_CFG} | less\`

 Tip: To view the gory details of the tripwire database
 Run: \`twprint --print-dbfile${NONSTD_TW_CFG}  | less\`

 Tip: Check the encryption status of tripwire files
 Run: \`twadmin --examine${NONSTD_TW_CFG} ${DBFILE}\`
      \`twadmin --examine${NONSTD_TW_CFG} ${POLFILE}\`
      \`twadmin --examine${NONSTD_TW_CFG} ${TW_CFG}\`
"

# Routines that present an opportunity to test and run tripwire
# -------------------------------------------------------------

echo " Do you want to send a test e-mail to root@localhost, and"
printf " perform a tripwire scan of the filesystem now? [y/N] "
read -n 1 -t 60 RUN_TW_TESTS
echo
if [ "${RUN_TW_TESTS^}" == "Y" ]; then
  echo
  echo " Testing tripwire."
  echo " Sending a test email to root@localhost, then"
  echo " running \`tripwire --check\`"
  echo
  tripwire --test --email root@localhost
  tripwire --check --email-report ${NONSTD_TW_CFG}
else
  echo
  echo " To test the set-up of tripwire, run these commands:"
  echo
  echo " \`tripwire --test  --email root@localhost\`"
  echo " \`tripwire --check --email-report${NONSTD_TW_CFG}\`"
fi

if [ ! -f "${DBFILE}.bak" ]; then
  echo
  echo " Don't be alarmed by an \"added file violation\"."
  printf " Press a key when you are ready to continue ... "
  read -n 1 READY_TO_CONTINUE
  echo "
 \"Added file violations\" are expected on a new install of tripwire.
 Getting tripwire to be clear of violations in its database directory
 takes a few steps:  --init, --check, --update, --check, --update

 On a virgin install of tripwire, the first \`tripwire --init\` creates
 ${DBFILE}, and the first \`tripwire --check\` flags it
 as \"Added\".  On the first \`tripwire --update\` (or the second
 \`tripwire --init\`), tripwire adds ${DBFILE}.bak

 AFTER the first \`tripwire --check\`, update tripwire's database.
 Running \`tripwire --update\` requires passing a report name.
 Use an alias to ease the chore of obtaining the correct report name.

 Run:  \`alias last.tw.report='echo \$(ls -t ${REPORTDIR}/* | head -1)'\`
 Then: \`tripwire --update -r \$(last.tw.report)${NONSTD_TW_CFG}\`

 AFTER the first \`tripwire --update\`, make tripwire aware of its own
 backup database file.  Run \`tripwire --check\` to create a report that
 shows the \"added file violation\" due to the backup database file,
 then \`tripwire --update -r \$(last.tw.report)${NONSTD_TW_CFG}\`.
 Tripwire will add no more files to the ${DBDIR} directory.
"
fi
}

# ===================================================================
# "tripwire_exist_error" and "recite_ver" and "recite_help" messages
# ===================================================================

tripwire_exist_error ()
{
echo "
 This script, ${0##*/}, has no function aside from tripwire.
 On a Gentoo system, \`emerge tripwire\`

 Goodbye.
"
exit 2
}

recite_ver ()
{
echo "
 This is ${0##*/} version $VERSION
 A script to facilitate set-up of Tripwire
 https://sourceforge.net/projects/mktwpol
"
}

recite_help ()
{
recite_ver
echo " Usage: ${0##*/} [-c tw_cfg_dir] [-d db_root] [-r] [-u]
		   [-p \"tw-policy-generator\"|-b] [-h|-V]

	-c Put tripwire configuration files in \"tw_cfg_dir\"
	   Default tw_cfg_dir = /etc/tripwire
	-d Database and report directories go in a directory tree off \"db_root\"
	   Default db_root = /var/lib
	   Default tripwire database directory = /var/lib/tripwire
	   Default tripwire reports directory  = /var/lib/tripwire/report
	-r Remove twcfg.txt and twpol.txt after processing
	-u Skip generating tripwire keys and encrypted config
	-p Plain-text policy generator command line (escape with '\"' chars)
	   Default policy text generator command line is \"mktwpol.sh\"
	-b Bypass plain-text policy generator
	   Use most recent \"twpol*txt\" file for tripwire policy
	-h Show version and help information
	-V Show version information

 Makes \"tw_cfg_dir\" directory if it does not exist
 Makes database and report directories off \"db_root\" if they do not exist
 Can configure tripwire into an empty directory
"
exit
}

#  =======   End of Subroutines   =======

# ========================================
# Main Routine
# ============

# Temporary (?!) code to facilitate debugging individual routines
[ "$1" == "debug" ] && { eval $2 ; exit ; }
[[ "${*}" =~ "null" ]] && NULL_PASSPHRASE=Y

while getopts :c:d:R:rup:bhV OPTION
do
  case $OPTION in
    c	) TW_CFG=$OPTARG	;;
    d	) DB_ROOT=$OPTARG	;;
    R	) RULES_FILE=$OPTARG	;;
    r	) AUTO_RM=Yes		;;
    u	) AUTO_UPDATE=Yes	;;
    p	) MKTWPOL_CMD_LINE="$OPTARG"	;;
    b	) BYPASS_GENERATOR=Yes	;;
    h	) recite_help		;;
    V	) recite_ver; exit	;;
    *	) recite_help		;;
  esac

done
shift $(($OPTIND - 1))

hash tripwire 2> /dev/null || tripwire_exist_error
hash twadmin  2> /dev/null || tripwire_exist_error

recite_ver
config_twsetup
if [ "${AUTO_UPDATE}" != "Yes" -o ! -r ${TW_CFG} ] ;then
  introduce_twsetup
  confirm_twcfg
  check_db_report_dirs
  make_key_files
  sign_twcfg_file
else
  echo " twsetup.sh invoked in UPDATE mode.  Skipping steps 1-4."
fi
make_plain_text_policy
encrypt_policy
scan_filesystem
if [ "${AUTO_UPDATE}" != "Yes" ] ;then
  make_cronjob
  outro_twsetup
fi
