(** Bindings of a C library *)type gnutls_cipher_algorithm_t
type gnutls_kx_algorithm_t
type gnutls_mac_algorithm_t
type gnutls_compression_method_t
type gnutls_pk_algorithm_t
type gnutls_sign_algorithm_t
type gnutls_certificate_type_t
type gnutls_digest_algorithm_t = gnutls_mac_algorithm_t
type gnutls_session_t
type gnutls_dh_params_t
type gnutls_priority_t
type gnutls_cipher_hd_t
type gnutls_x509_privkey_t
type gnutls_x509_crl_t
type gnutls_x509_crt_t
type gnutls_certificate_credentials_t
type gnutls_anon_server_credentials_t
type gnutls_anon_client_credentials_t
type gnutls_srp_server_credentials_t
type gnutls_srp_client_credentials_t
type gnutls_psk_server_credentials_t
type gnutls_psk_client_credentials_t
type gnutls_pubkey_t
type gnutls_privkey_t
type gnutls_params_type_t =
  [ `Rsa_export
  | `Dh
  | `Ecdh
  ]
type gnutls_credentials_type_t =
  [ `Certificate
  | `Anon
  | `Srp
  | `Psk
  | `Ia
  ]
type gnutls_alert_level_t =
  [ `Warning
  | `Fatal
  ]
type gnutls_alert_description_t =
  [ `Close_notify
  | `Unexpected_message
  | `Bad_record_mac
  | `Decryption_failed
  | `Record_overflow
  | `Decompression_failure
  | `Handshake_failure
  | `Ssl3_no_certificate
  | `Bad_certificate
  | `Unsupported_certificate
  | `Certificate_revoked
  | `Certificate_expired
  | `Certificate_unknown
  | `Illegal_parameter
  | `Unknown_ca
  | `Access_denied
  | `Decode_error
  | `Decrypt_error
  | `Export_restriction
  | `Protocol_version
  | `Insufficient_security
  | `Internal_error
  | `User_canceled
  | `No_renegotiation
  | `Unsupported_extension
  | `Certificate_unobtainable
  | `Unrecognized_name
  | `Unknown_psk_identity
  | `No_application_protocol
  | `Inner_application_failure
  | `Inner_application_verification
  ]
type gnutls_handshake_description_t =
  [ `Hello_request
  | `Client_hello
  | `Server_hello
  | `Hello_verify_request
  | `New_session_ticket
  | `Certificate_pkt
  | `Server_key_exchange
  | `Certificate_request
  | `Server_hello_done
  | `Certificate_verify
  | `Client_key_exchange
  | `Finished
  | `Supplemental
  | `Change_cipher_spec
  | `Client_hello_v2
  ]
type gnutls_certificate_status_t_flag =
  [ `Invalid
  | `Revoked
  | `Signer_not_found
  | `Signer_not_ca
  | `Insecure_algorithm
  | `Not_activated
  | `Expired
  | `Signature_failure
  | `Revocation_data_superseded
  | `Unexpected_owner
  | `Revocation_data_issued_in_future
  | `Signer_constraints_failure
  | `Mismatch
  | `Purpose_mismatch
  ]
type gnutls_certificate_status_t = gnutls_certificate_status_t_flag list
type gnutls_certificate_request_t =
  [ `Ignore
  | `Request
  | `Require
  ]
type gnutls_certificate_verify_flags_flag =
  [ `Disable_ca_sign
  | `Allow_x509_v1_ca_crt
  | `Do_not_allow_same
  | `Allow_any_x509_v1_ca_crt
  | `Allow_sign_rsa_md2
  | `Allow_sign_rsa_md5
  | `Disable_time_checks
  | `Disable_trusted_time_checks
  | `Do_not_allow_x509_v1_ca_crt
  | `Disable_crl_checks
  ]
type gnutls_certificate_verify_flags = gnutls_certificate_verify_flags_flag list
type gnutls_close_request_t =
  [ `Rdwr
  | `Wr
  ]
type gnutls_protocol_t =
  [ `Ssl3
  | `Tls1_0
  | `Tls1_1
  | `Tls1_2
  | `Dtls0_9
  | `Dtls1_0
  | `Dtls1_2
  | `Version_unknown
  ]
type gnutls_x509_crt_fmt_t =
  [ `Der
  | `Pem
  ]
type gnutls_certificate_print_formats_t =
  [ `Full
  | `Oneline
  | `Unsigned_full
  | `Compact
  | `Full_numbers
  ]
type gnutls_ecc_curve_t =
  [ `Invalid
  | `Secp224r1
  | `Secp256r1
  | `Secp384r1
  | `Secp521r1
  | `Secp192r1
  ]
type gnutls_sec_param_t =
  [ `Unknown
  | `Weak
  | `Low
  | `Normal
  | `High
  | `Ultra
  | `Insecure
  | `Export
  | `Very_weak
  | `Legacy
  ]
type gnutls_init_flags_flag =
  [ `Server
  | `Client
  | `Datagram
  | `Nonblock
  | `No_extensions
  | `No_replay_protection
  ]
type gnutls_init_flags = gnutls_init_flags_flag list
type gnutls_server_name_type_t =
  [ `Dns
  ]
type gnutls_supplemental_data_format_type_t =
  [ `Unknown
  ]
type gnutls_psk_key_flags =
  [ `Raw
  | `Hex
  ]
type gnutls_x509_subject_alt_name_t =
  [ `Dnsname
  | `Rfc822name
  | `Uri
  | `Ipaddress
  | `Othername
  | `Dn
  | `Othername_xmpp
  ]
type gnutls_privkey_type_t =
  [ `X509
  | `Openpgp
  | `Pkcs11
  | `Ext
  ]
type key_usage_flag =
  [ `Digital_signature
  | `Non_repudiation
  | `Key_encipherment
  | `Data_encipherment
  | `Key_agreement
  | `Key_cert_sign
  | `Crl_sign
  | `Encipher_only
  | `Decipher_only
  ]
type key_usage = key_usage_flag list
type gnutls_channel_binding_t =
  [ `Tls_unique
  ]
type gnutls_info_access_what_t =
  [ `Accessmethod_oid
  | `Accesslocation_generalname_type
  | `Uri
  | `Ocsp_uri
  | `Caissuers_uri
  | `Unknown
  ]
type gnutls_certificate_import_flags_flag =
  [ `Fail_if_unsorted
  | `Import_fail_if_exceed
  | `Sort
  ]
type gnutls_certificate_import_flags = gnutls_certificate_import_flags_flag list
type error_code =
  [ `Success
  | `Unknown_compression_algorithm
  | `Unknown_cipher_type
  | `Large_packet
  | `Unsupported_version_packet
  | `Unexpected_packet_length
  | `Invalid_session
  | `Fatal_alert_received
  | `Unexpected_packet
  | `Warning_alert_received
  | `Error_in_finished_packet
  | `Unexpected_handshake_packet
  | `Unknown_cipher_suite
  | `Unwanted_algorithm
  | `Mpi_scan_failed
  | `Decryption_failed
  | `Memory_error
  | `Decompression_failed
  | `Compression_failed
  | `Again
  | `Expired
  | `Db_error
  | `Srp_pwd_error
  | `Insufficient_credentials
  | `Hash_failed
  | `Base64_decoding_error
  | `Mpi_print_failed
  | `Rehandshake
  | `Got_application_data
  | `Record_limit_reached
  | `Encryption_failed
  | `Pk_encryption_failed
  | `Pk_decryption_failed
  | `Pk_sign_failed
  | `X509_unsupported_critical_extension
  | `Key_usage_violation
  | `No_certificate_found
  | `Invalid_request
  | `Short_memory_buffer
  | `Interrupted
  | `Push_error
  | `Pull_error
  | `Received_illegal_parameter
  | `Requested_data_not_available
  | `Pkcs1_wrong_pad
  | `Received_illegal_extension
  | `Internal_error
  | `Dh_prime_unacceptable
  | `File_error
  | `Too_many_empty_packets
  | `Unknown_pk_algorithm
  | `Too_many_handshake_packets
  | `No_temporary_rsa_params
  | `No_compression_algorithms
  | `No_cipher_suites
  | `Openpgp_getkey_failed
  | `Pk_sig_verify_failed
  | `Illegal_srp_username
  | `Srp_pwd_parsing_error
  | `No_temporary_dh_params
  | `Asn1_element_not_found
  | `Asn1_identifier_not_found
  | `Asn1_der_error
  | `Asn1_value_not_found
  | `Asn1_generic_error
  | `Asn1_value_not_valid
  | `Asn1_tag_error
  | `Asn1_tag_implicit
  | `Asn1_type_any_error
  | `Asn1_syntax_error
  | `Asn1_der_overflow
  | `Openpgp_uid_revoked
  | `Certificate_error
  | `Certificate_key_mismatch
  | `Unsupported_certificate_type
  | `X509_unknown_san
  | `Openpgp_fingerprint_unsupported
  | `X509_unsupported_attribute
  | `Unknown_hash_algorithm
  | `Unknown_pkcs_content_type
  | `Unknown_pkcs_bag_type
  | `Invalid_password
  | `Mac_verify_failed
  | `Constraint_error
  | `Warning_ia_iphf_received
  | `Warning_ia_fphf_received
  | `Ia_verify_failed
  | `Unknown_algorithm
  | `Unsupported_signature_algorithm
  | `Safe_renegotiation_failed
  | `Unsafe_renegotiation_denied
  | `Unknown_srp_username
  | `Premature_termination
  | `Base64_encoding_error
  | `Incompatible_crypto_library
  | `Incompatible_libtasn1_library
  | `Openpgp_keyring_error
  | `X509_unsupported_oid
  | `Random_failed
  | `Base64_unexpected_header_error
  | `Openpgp_subkey_error
  | `Crypto_already_registered
  | `Handshake_too_large
  | `Cryptodev_ioctl_error
  | `Cryptodev_device_error
  | `Channel_binding_not_available
  | `Bad_cookie
  | `Openpgp_preferred_key_error
  | `Incompat_dsa_key_with_tls_protocol
  | `Heartbeat_pong_received
  | `Heartbeat_ping_received
  | `Pkcs11_error
  | `Pkcs11_load_error
  | `Parsing_error
  | `Pkcs11_pin_error
  | `Pkcs11_slot_error
  | `Locking_error
  | `Pkcs11_attribute_error
  | `Pkcs11_device_error
  | `Pkcs11_data_error
  | `Pkcs11_unsupported_feature_error
  | `Pkcs11_key_error
  | `Pkcs11_pin_expired
  | `Pkcs11_pin_locked
  | `Pkcs11_session_error
  | `Pkcs11_signature_error
  | `Pkcs11_token_error
  | `Pkcs11_user_error
  | `Crypto_init_failed
  | `Timedout
  | `User_error
  | `Ecc_no_supported_curves
  | `Ecc_unsupported_curve
  | `Pkcs11_requested_object_not_availble
  | `Certificate_list_unsorted
  | `Illegal_parameter
  | `No_priorities_were_set
  | `X509_unsupported_extension
  | `Session_eof
  | `Tpm_error
  | `Tpm_key_password_error
  | `Tpm_srk_password_error
  | `Tpm_session_error
  | `Tpm_key_not_found
  | `Tpm_uninitialized
  | `No_certificate_status
  | `Ocsp_response_error
  | `Random_device_error
  | `Auth_error
  | `No_application_protocol
  | `Sockets_init_error
  | `Key_import_failed
  | `Self_test_error
  | `No_self_test
  | `Lib_in_error_state
  | `Pk_generation_error
  | `Idna_error
  | `Need_fallback
  | `Unimplemented_feature
  ]
type gnutls_pkcs_encrypt_flags_t_flag =
  [ `Plain
  | `Use_pkcs12_3des
  | `Use_pkcs12_arcfour
  | `Use_pkcs12_rc2_40
  | `Use_pbes2_3des
  | `Use_pbes2_des
  | `Use_pbes2_aes_128
  | `Use_pbes2_aes_192
  | `Use_pbes2_aes_256
  | `Null_password
  ]
type gnutls_pkcs_encrypt_flags_t = gnutls_pkcs_encrypt_flags_t_flag list
type empty_flags_flag =
  [ `Dummy
  ]
type empty_flags = empty_flags_flag list
type str_datum = string
type str_datum_p = string
type const_str_datum_p = string
val nettls_init : unit -> unit
val nettls_deinit : unit -> unit
val gnutls_error_is_fatal : error_code -> bool
val gnutls_strerror : error_code -> string
val gnutls_strerror_name : error_code -> string
val gnutls_check_version : unit -> string
val gnutls_global_set_log_level : int -> unit
val gnutls_init : gnutls_init_flags -> gnutls_session_t
val gnutls_certificate_server_set_request : gnutls_session_t -> gnutls_certificate_request_t -> unit
val gnutls_certificate_send_x509_rdn_sequence : gnutls_session_t -> bool -> unit
val gnutls_certificate_verify_peers2 : gnutls_session_t -> gnutls_certificate_status_t
val gnutls_certificate_get_peers : gnutls_session_t -> string array
val gnutls_record_get_direction : gnutls_session_t -> bool
val gnutls_handshake : gnutls_session_t -> unit
val gnutls_rehandshake : gnutls_session_t -> unit
val gnutls_record_send : gnutls_session_t -> Netsys_mem.memory -> int -> int
val gnutls_record_recv : gnutls_session_t -> Netsys_mem.memory -> int
val gnutls_record_check_pending : gnutls_session_t -> int
val gnutls_bye : gnutls_session_t -> gnutls_close_request_t -> unit
val gnutls_alert_get : gnutls_session_t -> gnutls_alert_description_t
val gnutls_alert_get_name : gnutls_alert_description_t -> string
val gnutls_alert_get_strname : gnutls_alert_description_t -> string
val gnutls_alert_send : gnutls_session_t -> gnutls_alert_level_t -> gnutls_alert_description_t -> unit
val gnutls_alert_send_appropriate : gnutls_session_t -> int -> unit
val gnutls_error_to_alert : error_code -> gnutls_alert_description_t * gnutls_alert_level_t
val gnutls_priority_init : string -> gnutls_priority_t
val gnutls_priority_get_cipher_suite_index : gnutls_priority_t -> int -> int
val gnutls_priority_set : gnutls_session_t -> gnutls_priority_t -> unit
val gnutls_set_default_priority : gnutls_session_t -> unit
val gnutls_sec_param_to_pk_bits : gnutls_pk_algorithm_t -> gnutls_sec_param_t -> int
val gnutls_pk_bits_to_sec_param : gnutls_pk_algorithm_t -> int -> gnutls_sec_param_t
val gnutls_sec_param_get_name : gnutls_sec_param_t -> string
val gnutls_session_set_data : gnutls_session_t -> Bytes.t -> unit
val gnutls_session_get_data : gnutls_session_t -> Bytes.t
val gnutls_session_get_id : gnutls_session_t -> Bytes.t
val gnutls_session_is_resumed : gnutls_session_t -> bool
val gnutls_db_set_cache_expiration : gnutls_session_t -> int -> unit
val gnutls_db_remove_session : gnutls_session_t -> unit
val gnutls_db_check_entry : gnutls_session_t -> string -> int
val gnutls_session_ticket_key_generate : unit -> string
val gnutls_session_ticket_enable_client : gnutls_session_t -> unit
val gnutls_session_ticket_enable_server : gnutls_session_t -> str_datum_p -> unit
val gnutls_prf : gnutls_session_t -> Netsys_mem.memory -> bool -> Netsys_mem.memory -> Netsys_mem.memory -> unit
val gnutls_prf_raw : gnutls_session_t -> Netsys_mem.memory -> Netsys_mem.memory -> Netsys_mem.memory -> unit
val gnutls_session_channel_binding : gnutls_session_t -> gnutls_channel_binding_t -> string
val gnutls_ecc_curve_get_name : gnutls_ecc_curve_t -> string
val gnutls_ecc_curve_get_size : gnutls_ecc_curve_t -> int
val gnutls_ecc_curve_get : gnutls_session_t -> gnutls_ecc_curve_t
val gnutls_cipher_get : gnutls_session_t -> gnutls_cipher_algorithm_t
val gnutls_kx_get : gnutls_session_t -> gnutls_kx_algorithm_t
val gnutls_mac_get : gnutls_session_t -> gnutls_mac_algorithm_t
val gnutls_compression_get : gnutls_session_t -> gnutls_compression_method_t
val gnutls_certificate_type_get : gnutls_session_t -> gnutls_certificate_type_t
val gnutls_cipher_get_key_size : gnutls_cipher_algorithm_t -> int
val gnutls_mac_get_key_size : gnutls_mac_algorithm_t -> int
val gnutls_cipher_get_name : gnutls_cipher_algorithm_t -> string
val gnutls_mac_get_name : gnutls_mac_algorithm_t -> string
val gnutls_compression_get_name : gnutls_compression_method_t -> string
val gnutls_kx_get_name : gnutls_kx_algorithm_t -> string
val gnutls_certificate_type_get_name : gnutls_certificate_type_t -> string
val gnutls_pk_get_name : gnutls_pk_algorithm_t -> string
val gnutls_sign_get_name : gnutls_sign_algorithm_t -> string
val gnutls_mac_get_id : string -> gnutls_mac_algorithm_t
val gnutls_compression_get_id : string -> gnutls_compression_method_t
val gnutls_cipher_get_id : string -> gnutls_cipher_algorithm_t
val gnutls_kx_get_id : string -> gnutls_kx_algorithm_t
val gnutls_protocol_get_id : string -> gnutls_protocol_t
val gnutls_certificate_type_get_id : string -> gnutls_certificate_type_t
val gnutls_pk_get_id : string -> gnutls_pk_algorithm_t
val gnutls_sign_get_id : string -> gnutls_sign_algorithm_t
val gnutls_cipher_list : unit -> gnutls_cipher_algorithm_t list
val gnutls_mac_list : unit -> gnutls_mac_algorithm_t list
val gnutls_compression_list : unit -> gnutls_compression_method_t list
val gnutls_protocol_list : unit -> gnutls_protocol_t list
val gnutls_certificate_type_list : unit -> gnutls_certificate_type_t list
val gnutls_kx_list : unit -> gnutls_kx_algorithm_t list
val gnutls_pk_list : unit -> gnutls_pk_algorithm_t list
val gnutls_sign_list : unit -> gnutls_sign_algorithm_t list
val gnutls_protocol_get_version : gnutls_session_t -> gnutls_protocol_t
val gnutls_protocol_get_name : gnutls_protocol_t -> string
val gnutls_cipher_suite_get_name : gnutls_kx_algorithm_t -> gnutls_cipher_algorithm_t -> gnutls_mac_algorithm_t -> string
val net_have_crypto : unit -> bool
val gnutls_cipher_init : gnutls_cipher_algorithm_t -> str_datum_p -> str_datum_p -> gnutls_cipher_hd_t
val gnutls_cipher_encrypt2 : gnutls_cipher_hd_t -> Netsys_mem.memory -> Netsys_mem.memory -> unit
val gnutls_cipher_decrypt2 : gnutls_cipher_hd_t -> Netsys_mem.memory -> Netsys_mem.memory -> unit
val gnutls_cipher_add_auth : gnutls_cipher_hd_t -> Bytes.t -> unit
val gnutls_cipher_tag : gnutls_cipher_hd_t -> Bytes.t -> unit
val gnutls_handshake_set_private_extensions : gnutls_session_t -> bool -> unit
val gnutls_handshake_get_last_out : gnutls_session_t -> gnutls_handshake_description_t
val gnutls_handshake_get_last_in : gnutls_session_t -> gnutls_handshake_description_t
val gnutls_handshake_set_max_packet_length : gnutls_session_t -> int -> unit
val gnutls_sign_algorithm_get_requested : gnutls_session_t -> int -> gnutls_sign_algorithm_t
val gnutls_session_enable_compatibility_mode : gnutls_session_t -> unit
val gnutls_record_disable_padding : gnutls_session_t -> unit
val gnutls_record_get_max_size : gnutls_session_t -> int
val gnutls_record_set_max_size : gnutls_session_t -> int -> unit
val gnutls_server_name_set : gnutls_session_t -> gnutls_server_name_type_t -> Bytes.t -> unit
val gnutls_server_name_get : gnutls_session_t -> int -> Bytes.t * gnutls_server_name_type_t
val gnutls_safe_renegotiation_status : gnutls_session_t -> bool
val gnutls_supplemental_get_name : gnutls_supplemental_data_format_type_t -> string
val gnutls_credentials_clear : gnutls_session_t -> unit
val gnutls_key_generate : int -> string
val gnutls_fingerprint : gnutls_digest_algorithm_t -> str_datum_p -> Bytes.t
val gnutls_certificate_allocate_credentials : unit -> gnutls_certificate_credentials_t
val gnutls_certificate_set_x509_system_trust : gnutls_certificate_credentials_t -> unit
val gnutls_certificate_set_x509_trust_file : gnutls_certificate_credentials_t -> string -> gnutls_x509_crt_fmt_t -> unit
val gnutls_certificate_set_x509_trust_mem : gnutls_certificate_credentials_t -> str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_certificate_set_x509_crl_file : gnutls_certificate_credentials_t -> string -> gnutls_x509_crt_fmt_t -> unit
val gnutls_certificate_set_x509_crl_mem : gnutls_certificate_credentials_t -> str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_certificate_set_x509_key_file : gnutls_certificate_credentials_t -> string -> string -> gnutls_x509_crt_fmt_t -> unit
val gnutls_certificate_set_x509_key_mem : gnutls_certificate_credentials_t -> str_datum_p -> str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_certificate_set_x509_simple_pkcs12_file : gnutls_certificate_credentials_t -> string -> gnutls_x509_crt_fmt_t -> string -> unit
val gnutls_certificate_set_x509_simple_pkcs12_mem : gnutls_certificate_credentials_t -> str_datum_p -> gnutls_x509_crt_fmt_t -> string -> unit
val gnutls_certificate_set_x509_key : gnutls_certificate_credentials_t -> gnutls_x509_crt_t array -> gnutls_x509_privkey_t -> unit
val gnutls_certificate_set_x509_trust : gnutls_certificate_credentials_t -> gnutls_x509_crt_t array -> int
val gnutls_certificate_set_x509_crl : gnutls_certificate_credentials_t -> gnutls_x509_crl_t array -> int
val gnutls_anon_allocate_server_credentials : unit -> gnutls_anon_server_credentials_t
val gnutls_anon_allocate_client_credentials : unit -> gnutls_anon_client_credentials_t
val gnutls_dh_params_init : unit -> gnutls_dh_params_t
val gnutls_dh_params_import_raw : gnutls_dh_params_t -> str_datum_p -> str_datum_p -> unit
val gnutls_dh_params_import_pkcs3 : gnutls_dh_params_t -> str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_dh_params_generate2 : gnutls_dh_params_t -> int -> unit
val gnutls_dh_params_cpy : gnutls_dh_params_t -> gnutls_dh_params_t -> unit
val gnutls_certificate_set_dh_params : gnutls_certificate_credentials_t -> gnutls_dh_params_t -> unit
val gnutls_anon_set_server_dh_params : gnutls_anon_server_credentials_t -> gnutls_dh_params_t -> unit
val gnutls_psk_set_server_dh_params : gnutls_psk_server_credentials_t -> gnutls_dh_params_t -> unit
val gnutls_dh_set_prime_bits : gnutls_session_t -> int -> unit
val gnutls_dh_get_secret_bits : gnutls_session_t -> int
val gnutls_dh_get_peers_public_bits : gnutls_session_t -> int
val gnutls_dh_get_prime_bits : gnutls_session_t -> int
val gnutls_dh_get_group : gnutls_session_t -> string * string
val gnutls_dh_get_pubkey : gnutls_session_t -> string
val gnutls_auth_get_type : gnutls_session_t -> gnutls_credentials_type_t
val gnutls_auth_server_get_type : gnutls_session_t -> gnutls_credentials_type_t
val gnutls_auth_client_get_type : gnutls_session_t -> gnutls_credentials_type_t
val gnutls_certificate_get_ours : gnutls_session_t -> const_str_datum_p
val gnutls_certificate_activation_time_peers : gnutls_session_t -> float
val gnutls_certificate_expiration_time_peers : gnutls_session_t -> float
val gnutls_certificate_client_get_request_status : gnutls_session_t -> bool
val gnutls_certificate_get_issuer : gnutls_certificate_credentials_t -> gnutls_x509_crt_t -> empty_flags -> gnutls_x509_crt_t
val gnutls_certificate_set_verify_flags : gnutls_certificate_credentials_t -> gnutls_certificate_verify_flags -> unit
val gnutls_certificate_set_verify_limits : gnutls_certificate_credentials_t -> int -> int -> unit
val gnutls_srp_allocate_client_credentials : unit -> gnutls_srp_client_credentials_t
val gnutls_srp_set_client_credentials : gnutls_srp_client_credentials_t -> string -> string -> unit
val gnutls_srp_allocate_server_credentials : unit -> gnutls_srp_server_credentials_t
val gnutls_srp_set_server_credentials_file : gnutls_srp_server_credentials_t -> string -> string -> unit
val gnutls_srp_server_get_username : gnutls_session_t -> string
val gnutls_srp_set_prime_bits : gnutls_session_t -> int -> unit
val gnutls_srp_verifier : string -> string -> str_datum_p -> str_datum_p -> str_datum_p -> string
val gnutls_psk_allocate_client_credentials : unit -> gnutls_psk_client_credentials_t
val gnutls_psk_set_client_credentials : gnutls_psk_client_credentials_t -> string -> str_datum_p -> gnutls_psk_key_flags -> unit
val gnutls_psk_allocate_server_credentials : unit -> gnutls_psk_server_credentials_t
val gnutls_psk_set_server_credentials_file : gnutls_psk_server_credentials_t -> string -> unit
val gnutls_psk_set_server_credentials_hint : gnutls_psk_server_credentials_t -> string -> unit
val gnutls_psk_server_get_username : gnutls_session_t -> string
val gnutls_psk_client_get_hint : gnutls_session_t -> string
val gnutls_x509_crt_init : unit -> gnutls_x509_crt_t
val gnutls_x509_crt_import : gnutls_x509_crt_t -> const_str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_x509_crt_export : gnutls_x509_crt_t -> gnutls_x509_crt_fmt_t -> Bytes.t
val gnutls_x509_crt_get_issuer_dn : gnutls_x509_crt_t -> Bytes.t
val gnutls_x509_crt_get_issuer_dn_oid : gnutls_x509_crt_t -> int -> Bytes.t
val gnutls_x509_crt_get_dn : gnutls_x509_crt_t -> Bytes.t
val gnutls_x509_crt_get_dn_oid : gnutls_x509_crt_t -> int -> Bytes.t
val gnutls_x509_crt_check_hostname : gnutls_x509_crt_t -> string -> bool
val gnutls_x509_crt_get_signature_algorithm : gnutls_x509_crt_t -> gnutls_sign_algorithm_t
val gnutls_x509_crt_get_signature : gnutls_x509_crt_t -> Bytes.t
val gnutls_x509_crt_get_version : gnutls_x509_crt_t -> int
val gnutls_x509_crt_get_key_id : gnutls_x509_crt_t -> empty_flags -> Bytes.t
val gnutls_x509_crt_get_authority_key_id : gnutls_x509_crt_t -> Bytes.t * bool
val gnutls_x509_crt_get_subject_key_id : gnutls_x509_crt_t -> Bytes.t * bool
val gnutls_x509_crt_get_subject_unique_id : gnutls_x509_crt_t -> Bytes.t
val gnutls_x509_crt_get_issuer_unique_id : gnutls_x509_crt_t -> Bytes.t
val gnutls_x509_crt_get_authority_info_access : gnutls_x509_crt_t -> int -> gnutls_info_access_what_t -> string * bool
val gnutls_x509_crt_get_activation_time : gnutls_x509_crt_t -> float
val gnutls_x509_crt_get_expiration_time : gnutls_x509_crt_t -> float
val gnutls_x509_crt_get_serial : gnutls_x509_crt_t -> Bytes.t
val gnutls_x509_crt_get_pk_algorithm : gnutls_x509_crt_t -> gnutls_pk_algorithm_t * int
val gnutls_x509_crt_get_pk_rsa_raw : gnutls_x509_crt_t -> string * string
val gnutls_x509_crt_get_pk_dsa_raw : gnutls_x509_crt_t -> string * string * string * string
val gnutls_x509_crt_get_subject_alt_name : gnutls_x509_crt_t -> int -> gnutls_x509_subject_alt_name_t * Bytes.t * bool
val gnutls_x509_crt_get_subject_alt_othername_oid : gnutls_x509_crt_t -> int -> gnutls_x509_subject_alt_name_t * Bytes.t
val gnutls_x509_crt_get_issuer_alt_name : gnutls_x509_crt_t -> int -> gnutls_x509_subject_alt_name_t * Bytes.t * bool
val gnutls_x509_crt_get_issuer_alt_othername_oid : gnutls_x509_crt_t -> int -> gnutls_x509_subject_alt_name_t * Bytes.t
val gnutls_x509_crt_get_ca_status : gnutls_x509_crt_t -> bool
val gnutls_x509_crt_get_basic_constraints : gnutls_x509_crt_t -> bool * bool * int
val gnutls_x509_crt_get_key_usage : gnutls_x509_crt_t -> key_usage * bool
val gnutls_x509_crt_get_extension_oid : gnutls_x509_crt_t -> int -> Bytes.t
val gnutls_x509_crt_get_extension_info : gnutls_x509_crt_t -> int -> Bytes.t * bool
val gnutls_x509_crt_get_extension_data : gnutls_x509_crt_t -> int -> Bytes.t
val gnutls_x509_crt_check_issuer : gnutls_x509_crt_t -> gnutls_x509_crt_t -> bool
val gnutls_x509_crt_list_verify : gnutls_x509_crt_t array -> gnutls_x509_crt_t array -> gnutls_x509_crl_t array -> gnutls_certificate_verify_flags -> gnutls_certificate_status_t
val gnutls_x509_crt_verify : gnutls_x509_crt_t -> gnutls_x509_crt_t array -> gnutls_certificate_verify_flags -> gnutls_certificate_status_t
val gnutls_x509_crl_verify : gnutls_x509_crl_t -> gnutls_x509_crt_t array -> gnutls_certificate_verify_flags -> gnutls_certificate_status_t
val gnutls_x509_crt_check_revocation : gnutls_x509_crt_t -> gnutls_x509_crl_t array -> bool
val gnutls_x509_crt_get_fingerprint : gnutls_x509_crt_t -> gnutls_digest_algorithm_t -> Bytes.t
val gnutls_x509_crt_get_key_purpose_oid : gnutls_x509_crt_t -> int -> Bytes.t * bool
val gnutls_x509_privkey_init : unit -> gnutls_x509_privkey_t
val gnutls_x509_privkey_import : gnutls_x509_privkey_t -> const_str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_x509_privkey_import_pkcs8 : gnutls_x509_privkey_t -> const_str_datum_p -> gnutls_x509_crt_fmt_t -> string -> gnutls_pkcs_encrypt_flags_t -> unit
val gnutls_x509_crl_init : unit -> gnutls_x509_crl_t
val gnutls_x509_crl_import : gnutls_x509_crl_t -> const_str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val net_have_pubkey : unit -> bool
val gnutls_pubkey_init : unit -> gnutls_pubkey_t
val gnutls_privkey_init : unit -> gnutls_privkey_t
val gnutls_pubkey_import : gnutls_pubkey_t -> const_str_datum_p -> gnutls_x509_crt_fmt_t -> unit
val gnutls_pubkey_import_url : gnutls_pubkey_t -> string -> int -> unit
val gnutls_pubkey_import_privkey : gnutls_pubkey_t -> gnutls_privkey_t -> int -> int -> unit
val gnutls_privkey_import_x509 : gnutls_privkey_t -> gnutls_x509_privkey_t -> int -> unit
val gnutls_pubkey_encrypt_data : gnutls_pubkey_t -> int -> const_str_datum_p -> string
val gnutls_privkey_decrypt_data : gnutls_privkey_t -> int -> const_str_datum_p -> string
val gnutls_privkey_sign_data : gnutls_privkey_t -> gnutls_digest_algorithm_t -> int -> const_str_datum_p -> string
val gnutls_pubkey_verify_data2 : gnutls_pubkey_t -> gnutls_sign_algorithm_t -> int -> const_str_datum_p -> const_str_datum_p -> unit
val b_error_of_name : string -> error_code
(* This file is included into nettls_gnutls_bindings.mli *)

exception Null_pointer
exception Error of error_code
exception Short_memory_buffer of int

type memory = 
    (char,Bigarray.int8_unsigned_elt,Bigarray.c_layout) Bigarray.Array1.t
  (** See {!Netsys_types.memory} *)

type gnutls_credentials =
    [ `Certificate of gnutls_certificate_credentials_t
    | `Srp_client of gnutls_srp_client_credentials_t
    | `Srp_server of gnutls_srp_server_credentials_t
    | `Psk_client of gnutls_psk_client_credentials_t
    | `Psk_server of gnutls_psk_server_credentials_t
    | `Anon_client of gnutls_anon_client_credentials_t
    | `Anon_server of gnutls_anon_server_credentials_t
    ]

val gnutls_credentials_set : gnutls_session_t -> gnutls_credentials -> unit

val b_set_pull_callback : 
  gnutls_session_t -> (memory -> int) -> unit
  (** Sets the function for reading data. The function must return the number
      of read bytes (like [Unix.read]). The function can raise [Unix_error].
      Only the codes [EINTR], [EAGAIN], [EWOULDBLOCK], and [EMSGSIZE] are
      interpreted.
   *)

val b_set_push_callback : 
  gnutls_session_t -> (memory -> int -> int) -> unit
  (** Sets the function for writing data. The function must return the number
      of written bytes (like [Unix.write]). The function can raise [Unix_error].
      Only the codes [EINTR], [EAGAIN], [EWOULDBLOCK], and [EMSGSIZE] are
      interpreted.
   *)

val b_set_pull_timeout_callback : 
  gnutls_session_t -> (int -> bool) -> unit
  (** Sets the function for waiting for new data to arrive (only used for
      DTLS). The integer are the milliseconds to wait at most. The function
      shall return [true] if there is data, and [false] in case of a timeout.
      The function can raise [Unix_error].
      Only the codes [EINTR], [EAGAIN], [EWOULDBLOCK], and [EMSGSIZE] are
      interpreted.
   *)

val b_set_verify_callback :
  gnutls_session_t -> (gnutls_session_t -> bool) -> unit
  (** Sets the function for verifying the peer's certificate. The function
      can return [true] if the certificate is acceptable, and [false] otherwise.
      Note that this callback is set in this binding on the session, and
      not on [gnutls_certificate_credentials_t].
   *)


val set_fd : gnutls_session_t -> Unix.file_descr -> unit
  (** Sets that this file descriptor is used for I/O. NB. This function just
      invokes [b_set_pull_callback], [b_set_push_callback], and
      [b_set_pull_timeout_callback] with the right argument functions.
   *)

val b_set_db_callbacks :
  gnutls_session_t ->
  (string -> string -> unit) ->
  (string -> unit) ->
  (string -> string) ->
  unit
  (** [b_set_db_callbacks session store remove retrieve]: sets the three
      callbacks for server-side session caching
   *)

val string_of_verification_status_flag : 
      gnutls_certificate_status_t_flag -> string
  (** Returns a string for the status code *)

val gnutls_x509_crt_list_import : string -> gnutls_x509_crt_fmt_t ->
                                  gnutls_certificate_import_flags ->
                                  gnutls_x509_crt_t array

val gnutls_x509_crl_list_import : string -> gnutls_x509_crt_fmt_t ->
                                  gnutls_certificate_import_flags ->
                                  gnutls_x509_crl_t array
